/**
 * JavaScript Personalizado - Sistema de Clínicas
 */

$(document).ready(function() {
    // Aplicar máscaras nos campos
    aplicarMascaras();
    
    // Configurar tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
    
    // Configurar popovers
    var popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
    var popoverList = popoverTriggerList.map(function (popoverTriggerEl) {
        return new bootstrap.Popover(popoverTriggerEl);
    });
    
    // Auto-hide alerts após 5 segundos
    setTimeout(function() {
        $('.alert').fadeOut('slow');
    }, 5000);
    
    // Confirmação para exclusões
    $('.btn-delete').on('click', function(e) {
        e.preventDefault();
        var url = $(this).attr('href');
        var item = $(this).data('item') || 'este item';
        
        if (confirm('Tem certeza que deseja excluir ' + item + '? Esta ação não pode ser desfeita.')) {
            window.location.href = url;
        }
    });
    
    // Upload de arquivos com drag and drop
    configurarUploadDragDrop();
    
    // Validação de formulários
    configurarValidacaoFormularios();
    
    // Busca em tempo real
    configurarBuscaTempoReal();
});

/**
 * Aplicar máscaras nos campos de entrada
 */
function aplicarMascaras() {
    // CPF
    $('.cpf-mask').mask('000.000.000-00', {
        reverse: true,
        placeholder: '___.___.___-__'
    });
    
    // CNPJ
    $('.cnpj-mask').mask('00.000.000/0000-00', {
        reverse: true,
        placeholder: '__.___.___/____-__'
    });
    
    // Telefone
    $('.telefone-mask').mask('(00) 00000-0000', {
        placeholder: '(__) _____-____'
    });
    
    // CEP
    $('.cep-mask').mask('00000-000', {
        placeholder: '_____-___'
    });
    
    // Dinheiro
    $('.money-mask').mask('#.##0,00', {
        reverse: true,
        placeholder: '0,00'
    });
    
    // Data
    $('.date-mask').mask('00/00/0000', {
        placeholder: '__/__/____'
    });
    
    // Hora
    $('.time-mask').mask('00:00', {
        placeholder: '__:__'
    });
}

/**
 * Configurar upload com drag and drop
 */
function configurarUploadDragDrop() {
    $('.upload-area').on('dragover', function(e) {
        e.preventDefault();
        $(this).addClass('dragover');
    });
    
    $('.upload-area').on('dragleave', function(e) {
        e.preventDefault();
        $(this).removeClass('dragover');
    });
    
    $('.upload-area').on('drop', function(e) {
        e.preventDefault();
        $(this).removeClass('dragover');
        
        var files = e.originalEvent.dataTransfer.files;
        var input = $(this).find('input[type="file"]')[0];
        
        if (input && files.length > 0) {
            input.files = files;
            $(input).trigger('change');
        }
    });
    
    // Mostrar nome do arquivo selecionado
    $('input[type="file"]').on('change', function() {
        var fileName = $(this).val().split('\\').pop();
        var label = $(this).siblings('.upload-label');
        
        if (fileName) {
            label.text('Arquivo selecionado: ' + fileName);
            label.addClass('text-success');
        } else {
            label.text('Clique aqui ou arraste um arquivo');
            label.removeClass('text-success');
        }
    });
}

/**
 * Configurar validação de formulários
 */
function configurarValidacaoFormularios() {
    // Validação de CPF
    $('.cpf-mask').on('blur', function() {
        var cpf = $(this).val().replace(/[^\d]/g, '');
        if (cpf.length === 11 && !validarCPF(cpf)) {
            $(this).addClass('is-invalid');
            $(this).siblings('.invalid-feedback').text('CPF inválido');
        } else {
            $(this).removeClass('is-invalid');
        }
    });
    
    // Validação de CNPJ
    $('.cnpj-mask').on('blur', function() {
        var cnpj = $(this).val().replace(/[^\d]/g, '');
        if (cnpj.length === 14 && !validarCNPJ(cnpj)) {
            $(this).addClass('is-invalid');
            $(this).siblings('.invalid-feedback').text('CNPJ inválido');
        } else {
            $(this).removeClass('is-invalid');
        }
    });
    
    // Validação de email
    $('input[type="email"]').on('blur', function() {
        var email = $(this).val();
        var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        
        if (email && !emailRegex.test(email)) {
            $(this).addClass('is-invalid');
            $(this).siblings('.invalid-feedback').text('Email inválido');
        } else {
            $(this).removeClass('is-invalid');
        }
    });
    
    // Validação de campos obrigatórios
    $('input[required], select[required], textarea[required]').on('blur', function() {
        if (!$(this).val()) {
            $(this).addClass('is-invalid');
            $(this).siblings('.invalid-feedback').text('Este campo é obrigatório');
        } else {
            $(this).removeClass('is-invalid');
        }
    });
}

/**
 * Configurar busca em tempo real
 */
function configurarBuscaTempoReal() {
    $('.search-input').on('keyup', function() {
        var searchTerm = $(this).val().toLowerCase();
        var targetTable = $(this).data('target');
        
        $(targetTable + ' tbody tr').each(function() {
            var rowText = $(this).text().toLowerCase();
            if (rowText.indexOf(searchTerm) === -1) {
                $(this).hide();
            } else {
                $(this).show();
            }
        });
    });
}

/**
 * Validar CPF
 */
function validarCPF(cpf) {
    if (cpf.length !== 11 || /^(\d)\1{10}$/.test(cpf)) {
        return false;
    }
    
    var soma = 0;
    var resto;
    
    for (var i = 1; i <= 9; i++) {
        soma += parseInt(cpf.substring(i - 1, i)) * (11 - i);
    }
    
    resto = (soma * 10) % 11;
    if (resto === 10 || resto === 11) resto = 0;
    if (resto !== parseInt(cpf.substring(9, 10))) return false;
    
    soma = 0;
    for (var i = 1; i <= 10; i++) {
        soma += parseInt(cpf.substring(i - 1, i)) * (12 - i);
    }
    
    resto = (soma * 10) % 11;
    if (resto === 10 || resto === 11) resto = 0;
    if (resto !== parseInt(cpf.substring(10, 11))) return false;
    
    return true;
}

/**
 * Validar CNPJ
 */
function validarCNPJ(cnpj) {
    if (cnpj.length !== 14) return false;
    
    // Elimina CNPJs inválidos conhecidos
    if (/^(\d)\1{13}$/.test(cnpj)) return false;
    
    var tamanho = cnpj.length - 2;
    var numeros = cnpj.substring(0, tamanho);
    var digitos = cnpj.substring(tamanho);
    var soma = 0;
    var pos = tamanho - 7;
    
    for (var i = tamanho; i >= 1; i--) {
        soma += numeros.charAt(tamanho - i) * pos--;
        if (pos < 2) pos = 9;
    }
    
    var resultado = soma % 11 < 2 ? 0 : 11 - soma % 11;
    if (resultado !== parseInt(digitos.charAt(0))) return false;
    
    tamanho = tamanho + 1;
    numeros = cnpj.substring(0, tamanho);
    soma = 0;
    pos = tamanho - 7;
    
    for (var i = tamanho; i >= 1; i--) {
        soma += numeros.charAt(tamanho - i) * pos--;
        if (pos < 2) pos = 9;
    }
    
    resultado = soma % 11 < 2 ? 0 : 11 - soma % 11;
    if (resultado !== parseInt(digitos.charAt(1))) return false;
    
    return true;
}

/**
 * Formatar moeda brasileira
 */
function formatarMoeda(valor) {
    return new Intl.NumberFormat('pt-BR', {
        style: 'currency',
        currency: 'BRL'
    }).format(valor);
}

/**
 * Formatar data brasileira
 */
function formatarData(data) {
    return new Date(data).toLocaleDateString('pt-BR');
}

/**
 * Formatar data e hora brasileira
 */
function formatarDataHora(dataHora) {
    return new Date(dataHora).toLocaleString('pt-BR');
}

/**
 * Mostrar loading em botão
 */
function mostrarLoading(botao) {
    var textoOriginal = $(botao).html();
    $(botao).data('texto-original', textoOriginal);
    $(botao).html('<span class="loading-spinner"></span> Carregando...');
    $(botao).prop('disabled', true);
}

/**
 * Esconder loading em botão
 */
function esconderLoading(botao) {
    var textoOriginal = $(botao).data('texto-original');
    $(botao).html(textoOriginal);
    $(botao).prop('disabled', false);
}

/**
 * Mostrar notificação toast
 */
function mostrarToast(titulo, mensagem, tipo = 'info') {
    var toastHtml = `
        <div class="toast align-items-center text-white bg-${tipo} border-0" role="alert" aria-live="assertive" aria-atomic="true">
            <div class="d-flex">
                <div class="toast-body">
                    <strong>${titulo}</strong><br>
                    ${mensagem}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
            </div>
        </div>
    `;
    
    // Criar container de toasts se não existir
    if (!$('#toast-container').length) {
        $('body').append('<div id="toast-container" class="toast-container position-fixed top-0 end-0 p-3"></div>');
    }
    
    var $toast = $(toastHtml);
    $('#toast-container').append($toast);
    
    var toast = new bootstrap.Toast($toast[0]);
    toast.show();
    
    // Remover toast após ser escondido
    $toast.on('hidden.bs.toast', function() {
        $(this).remove();
    });
}

/**
 * Confirmar ação com modal
 */
function confirmarAcao(titulo, mensagem, callback) {
    var modalHtml = `
        <div class="modal fade" id="modalConfirmacao" tabindex="-1" aria-labelledby="modalConfirmacaoLabel" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="modalConfirmacaoLabel">${titulo}</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        ${mensagem}
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="button" class="btn btn-primary" id="btnConfirmar">Confirmar</button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // Remover modal existente se houver
    $('#modalConfirmacao').remove();
    
    $('body').append(modalHtml);
    var modal = new bootstrap.Modal(document.getElementById('modalConfirmacao'));
    
    $('#btnConfirmar').on('click', function() {
        modal.hide();
        if (typeof callback === 'function') {
            callback();
        }
    });
    
    modal.show();
}

/**
 * Gerar link do WhatsApp
 */
function gerarLinkWhatsApp(telefone, mensagem = '') {
    var numeroLimpo = telefone.replace(/[^\d]/g, '');
    if (numeroLimpo.length === 11 && numeroLimpo.charAt(0) !== '0') {
        numeroLimpo = '55' + numeroLimpo;
    }
    
    var url = 'https://wa.me/' + numeroLimpo;
    if (mensagem) {
        url += '?text=' + encodeURIComponent(mensagem);
    }
    
    return url;
}
