<?php
/**
 * Configurações Gerais do Sistema
 * Sistema de Controle e Administração Financeira de Clínicas
 */

// Configurações de sessão
session_start();

// Configurações de timezone
date_default_timezone_set('America/Sao_Paulo');

// Configurações de upload
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB em bytes
define('UPLOAD_PATH_CLINICAS', __DIR__ . '/../uploads/documentos_clinica/');
define('UPLOAD_PATH_PROFISSIONAIS', __DIR__ . '/../uploads/documentos_profissionais/');

// Tipos de documentos permitidos
define('ALLOWED_FILE_TYPES', ['pdf']);

// Configurações de segurança
define('HASH_ALGORITHM', 'sha256');
define('SESSION_TIMEOUT', 3600); // 1 hora em segundos

// Configurações de paginação
define('ITEMS_PER_PAGE', 20);

// Configurações de formato de data
define('DATE_FORMAT', 'd/m/Y');
define('DATETIME_FORMAT', 'd/m/Y H:i');

// Tipos de documentos fiscais para clínicas
$tipos_documentos_clinica = [
    'contrato_social' => 'Contrato Social',
    'cnpj' => 'CNPJ',
    'certidao_negativa_civel' => 'Certidão Negativa Cível',
    'cndt' => 'Certidão Negativa de Débitos Trabalhistas (CNDT)',
    'regularidade_federal' => 'Prova de Regularidade Federal',
    'regularidade_estadual' => 'Prova de Regularidade Estadual',
    'regularidade_municipal' => 'Prova de Regularidade Municipal',
    'fgts' => 'Fundo de Garantia por Tempo de Serviço (FGTS)',
    'conselho_medicina' => 'Conselho Regional de Medicina',
    'conselho_psicologia' => 'Conselho Regional de Psicologia',
    'taxa_vistoria' => 'Taxa Relativa à Vistoria Anual',
    'planta_baixa' => 'Planta Baixa',
    'alvara' => 'Alvará',
    'contrato_locacao' => 'Contrato de Locação',
    'certificado_bombeiros' => 'Certificado de Aprovação do Bombeiros',
    'licenca_sanitaria' => 'Licença Sanitária',
    'certificado_calibracao_rz' => 'Certificado de Calibração RZ'
];

// Formas de pagamento
$formas_pagamento = [
    'dinheiro' => 'Dinheiro',
    'cartao_credito' => 'Cartão de Crédito',
    'cartao_debito' => 'Cartão de Débito',
    'pix' => 'PIX',
    'transferencia' => 'Transferência Bancária',
    'cheque' => 'Cheque'
];

// Função para verificar se o usuário está logado
function verificarLogin() {
    if (!isset($_SESSION['usuario_id'])) {
        header('Location: /sistema_clinicas/login.php');
        exit();
    }
    
    // Verificar timeout da sessão
    if (isset($_SESSION['ultimo_acesso']) && (time() - $_SESSION['ultimo_acesso'] > SESSION_TIMEOUT)) {
        session_destroy();
        header('Location: /sistema_clinicas/login.php?timeout=1');
        exit();
    }
    
    $_SESSION['ultimo_acesso'] = time();
}

// Função para verificar permissões de acesso
function verificarPermissao($perfil_necessario = null, $id_clinica_necessaria = null) {
    verificarLogin();
    
    if ($perfil_necessario && $_SESSION['perfil_acesso'] !== 'Administrador') {
        if ($_SESSION['perfil_acesso'] !== $perfil_necessario) {
            header('Location: /sistema_clinicas/acesso_negado.php');
            exit();
        }
    }
    
    if ($id_clinica_necessaria && $_SESSION['perfil_acesso'] !== 'Administrador') {
        if ($_SESSION['id_clinica'] != $id_clinica_necessaria) {
            header('Location: /sistema_clinicas/acesso_negado.php');
            exit();
        }
    }
}

// Função para formatar CPF
function formatarCPF($cpf) {
    return preg_replace("/(\d{3})(\d{3})(\d{3})(\d{2})/", "$1.$2.$3-$4", $cpf);
}

// Função para formatar CNPJ
function formatarCNPJ($cnpj) {
    return preg_replace("/(\d{2})(\d{3})(\d{3})(\d{4})(\d{2})/", "$1.$2.$3/$4-$5", $cnpj);
}

// Função para formatar telefone
function formatarTelefone($telefone) {
    $telefone = preg_replace('/[^0-9]/', '', $telefone);
    if (strlen($telefone) == 11) {
        return preg_replace("/(\d{2})(\d{5})(\d{4})/", "($1) $2-$3", $telefone);
    } elseif (strlen($telefone) == 10) {
        return preg_replace("/(\d{2})(\d{4})(\d{4})/", "($1) $2-$3", $telefone);
    }
    return $telefone;
}

// Função para validar upload de arquivo
function validarUpload($arquivo) {
    $erros = [];
    
    if ($arquivo['error'] !== UPLOAD_ERR_OK) {
        $erros[] = 'Erro no upload do arquivo.';
        return $erros;
    }
    
    if ($arquivo['size'] > MAX_FILE_SIZE) {
        $erros[] = 'Arquivo muito grande. Tamanho máximo: 5MB.';
    }
    
    $extensao = strtolower(pathinfo($arquivo['name'], PATHINFO_EXTENSION));
    if (!in_array($extensao, ALLOWED_FILE_TYPES)) {
        $erros[] = 'Tipo de arquivo não permitido. Apenas PDF.';
    }
    
    return $erros;
}
?>
