<?php
require_once __DIR__ . '/../../config/database.php';

class Agendamento {
    private $conn;
    private $table_name = "agendamentos";
    
    public $id_agendamento;
    public $id_clinica;
    public $id_paciente;
    public $id_profissional;
    public $data_hora_consulta;
    public $tipo_exame_agendado;
    public $status_atendimento;
    public $observacao_agendamento;
    
    public function __construct() {
        $this->conn = getDBConnection();
    }
    
    /**
     * Criar novo agendamento
     */
    public function criar() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET id_clinica=:id_clinica, id_paciente=:id_paciente, id_profissional=:id_profissional,
                      data_hora_consulta=:data_hora_consulta, tipo_exame_agendado=:tipo_exame_agendado,
                      observacao_agendamento=:observacao_agendamento";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->id_clinica = htmlspecialchars(strip_tags($this->id_clinica));
        $this->id_paciente = htmlspecialchars(strip_tags($this->id_paciente));
        $this->id_profissional = htmlspecialchars(strip_tags($this->id_profissional));
        $this->data_hora_consulta = htmlspecialchars(strip_tags($this->data_hora_consulta));
        $this->tipo_exame_agendado = htmlspecialchars(strip_tags($this->tipo_exame_agendado));
        $this->observacao_agendamento = htmlspecialchars(strip_tags($this->observacao_agendamento));
        
        // Bind dos valores
        $stmt->bindParam(":id_clinica", $this->id_clinica);
        $stmt->bindParam(":id_paciente", $this->id_paciente);
        $stmt->bindParam(":id_profissional", $this->id_profissional);
        $stmt->bindParam(":data_hora_consulta", $this->data_hora_consulta);
        $stmt->bindParam(":tipo_exame_agendado", $this->tipo_exame_agendado);
        $stmt->bindParam(":observacao_agendamento", $this->observacao_agendamento);
        
        if($stmt->execute()) {
            $this->id_agendamento = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    /**
     * Ler dados de um agendamento
     */
    public function lerUm() {
        $query = "SELECT a.*, p.nome_paciente, p.cpf, p.telefone_contato, p.renach,
                         pr.nome_completo as nome_profissional, pr.tipo_registro,
                         c.nome_clinica
                  FROM " . $this->table_name . " a
                  INNER JOIN pacientes p ON a.id_paciente = p.id_paciente
                  LEFT JOIN profissionais pr ON a.id_profissional = pr.id_profissional
                  INNER JOIN clinicas c ON a.id_clinica = c.id_clinica
                  WHERE a.id_agendamento = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_agendamento);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->id_clinica = $row['id_clinica'];
            $this->id_paciente = $row['id_paciente'];
            $this->id_profissional = $row['id_profissional'];
            $this->data_hora_consulta = $row['data_hora_consulta'];
            $this->tipo_exame_agendado = $row['tipo_exame_agendado'];
            $this->status_atendimento = $row['status_atendimento'];
            $this->observacao_agendamento = $row['observacao_agendamento'];
            return $row;
        }
        
        return false;
    }
    
    /**
     * Atualizar agendamento
     */
    public function atualizar() {
        $query = "UPDATE " . $this->table_name . " 
                  SET id_clinica=:id_clinica, id_paciente=:id_paciente, id_profissional=:id_profissional,
                      data_hora_consulta=:data_hora_consulta, tipo_exame_agendado=:tipo_exame_agendado,
                      status_atendimento=:status_atendimento, observacao_agendamento=:observacao_agendamento
                  WHERE id_agendamento=:id_agendamento";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->id_clinica = htmlspecialchars(strip_tags($this->id_clinica));
        $this->id_paciente = htmlspecialchars(strip_tags($this->id_paciente));
        $this->id_profissional = htmlspecialchars(strip_tags($this->id_profissional));
        $this->data_hora_consulta = htmlspecialchars(strip_tags($this->data_hora_consulta));
        $this->tipo_exame_agendado = htmlspecialchars(strip_tags($this->tipo_exame_agendado));
        $this->status_atendimento = htmlspecialchars(strip_tags($this->status_atendimento));
        $this->observacao_agendamento = htmlspecialchars(strip_tags($this->observacao_agendamento));
        $this->id_agendamento = htmlspecialchars(strip_tags($this->id_agendamento));
        
        // Bind dos valores
        $stmt->bindParam(":id_clinica", $this->id_clinica);
        $stmt->bindParam(":id_paciente", $this->id_paciente);
        $stmt->bindParam(":id_profissional", $this->id_profissional);
        $stmt->bindParam(":data_hora_consulta", $this->data_hora_consulta);
        $stmt->bindParam(":tipo_exame_agendado", $this->tipo_exame_agendado);
        $stmt->bindParam(":status_atendimento", $this->status_atendimento);
        $stmt->bindParam(":observacao_agendamento", $this->observacao_agendamento);
        $stmt->bindParam(":id_agendamento", $this->id_agendamento);
        
        return $stmt->execute();
    }
    
    /**
     * Deletar agendamento
     */
    public function deletar() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id_agendamento = ?";
        
        $stmt = $this->conn->prepare($query);
        $this->id_agendamento = htmlspecialchars(strip_tags($this->id_agendamento));
        $stmt->bindParam(1, $this->id_agendamento);
        
        return $stmt->execute();
    }
    
    /**
     * Listar agendamentos por data
     */
    public function listarPorData($data, $id_clinica = null) {
        $where = "WHERE DATE(a.data_hora_consulta) = ?";
        
        if($id_clinica) {
            $where .= " AND a.id_clinica = ?";
        }
        
        $query = "SELECT a.*, p.nome_paciente, p.cpf, p.telefone_contato, p.renach,
                         pr.nome_completo as nome_profissional, pr.tipo_registro,
                         c.nome_clinica
                  FROM " . $this->table_name . " a
                  INNER JOIN pacientes p ON a.id_paciente = p.id_paciente
                  LEFT JOIN profissionais pr ON a.id_profissional = pr.id_profissional
                  INNER JOIN clinicas c ON a.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY a.data_hora_consulta ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Listar agendamentos por período
     */
    public function listarPorPeriodo($data_inicio, $data_fim, $id_clinica = null) {
        $where = "WHERE DATE(a.data_hora_consulta) BETWEEN ? AND ?";
        
        if($id_clinica) {
            $where .= " AND a.id_clinica = ?";
        }
        
        $query = "SELECT a.*, p.nome_paciente, p.cpf, p.telefone_contato, p.renach,
                         pr.nome_completo as nome_profissional, pr.tipo_registro,
                         c.nome_clinica
                  FROM " . $this->table_name . " a
                  INNER JOIN pacientes p ON a.id_paciente = p.id_paciente
                  LEFT JOIN profissionais pr ON a.id_profissional = pr.id_profissional
                  INNER JOIN clinicas c ON a.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY a.data_hora_consulta ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data_inicio);
        $stmt->bindParam(2, $data_fim);
        
        if($id_clinica) {
            $stmt->bindParam(3, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Verificar disponibilidade de horário
     */
    public function verificarDisponibilidade($data_hora, $id_profissional, $id_agendamento = null) {
        $query = "SELECT id_agendamento FROM " . $this->table_name . " 
                  WHERE data_hora_consulta = ? AND id_profissional = ? AND status_atendimento != 'Cancelado'";
        
        if($id_agendamento) {
            $query .= " AND id_agendamento != ?";
        }
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data_hora);
        $stmt->bindParam(2, $id_profissional);
        
        if($id_agendamento) {
            $stmt->bindParam(3, $id_agendamento);
        }
        
        $stmt->execute();
        
        return $stmt->rowCount() === 0;
    }
    
    /**
     * Marcar presença
     */
    public function marcarPresenca() {
        $query = "UPDATE " . $this->table_name . " 
                  SET status_atendimento = 'Presente' 
                  WHERE id_agendamento = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_agendamento);
        
        return $stmt->execute();
    }
    
    /**
     * Marcar falta
     */
    public function marcarFalta() {
        $query = "UPDATE " . $this->table_name . " 
                  SET status_atendimento = 'Faltou' 
                  WHERE id_agendamento = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_agendamento);
        
        return $stmt->execute();
    }
    
    /**
     * Cancelar agendamento
     */
    public function cancelar() {
        $query = "UPDATE " . $this->table_name . " 
                  SET status_atendimento = 'Cancelado' 
                  WHERE id_agendamento = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_agendamento);
        
        return $stmt->execute();
    }
    
    /**
     * Obter agendamentos do mês para calendário
     */
    public function obterAgendamentosMes($ano, $mes, $id_clinica = null) {
        $where = "WHERE YEAR(a.data_hora_consulta) = ? AND MONTH(a.data_hora_consulta) = ?";
        
        if($id_clinica) {
            $where .= " AND a.id_clinica = ?";
        }
        
        $query = "SELECT DATE(a.data_hora_consulta) as data, COUNT(*) as total,
                         COUNT(CASE WHEN a.status_atendimento = 'Agendado' THEN 1 END) as agendados,
                         COUNT(CASE WHEN a.status_atendimento = 'Presente' THEN 1 END) as presentes,
                         COUNT(CASE WHEN a.status_atendimento = 'Faltou' THEN 1 END) as faltas
                  FROM " . $this->table_name . " a
                  " . $where . "
                  GROUP BY DATE(a.data_hora_consulta)
                  ORDER BY DATE(a.data_hora_consulta)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $ano);
        $stmt->bindParam(2, $mes);
        
        if($id_clinica) {
            $stmt->bindParam(3, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Listar pacientes faltantes
     */
    public function listarFaltantes($data_inicio = null, $data_fim = null, $id_clinica = null) {
        $where = "WHERE a.status_atendimento = 'Faltou'";
        
        if($data_inicio && $data_fim) {
            $where .= " AND DATE(a.data_hora_consulta) BETWEEN ? AND ?";
        }
        
        if($id_clinica) {
            $where .= " AND a.id_clinica = ?";
        }
        
        $query = "SELECT a.*, p.nome_paciente, p.cpf, p.telefone_contato,
                         pr.nome_completo as nome_profissional,
                         c.nome_clinica
                  FROM " . $this->table_name . " a
                  INNER JOIN pacientes p ON a.id_paciente = p.id_paciente
                  LEFT JOIN profissionais pr ON a.id_profissional = pr.id_profissional
                  INNER JOIN clinicas c ON a.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY a.data_hora_consulta DESC";
        
        $stmt = $this->conn->prepare($query);
        
        $param_index = 1;
        if($data_inicio && $data_fim) {
            $stmt->bindParam($param_index++, $data_inicio);
            $stmt->bindParam($param_index++, $data_fim);
        }
        
        if($id_clinica) {
            $stmt->bindParam($param_index, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Obter estatísticas de agendamentos
     */
    public function obterEstatisticas($data_inicio = null, $data_fim = null, $id_clinica = null) {
        $where = "WHERE 1=1";
        
        if($data_inicio && $data_fim) {
            $where .= " AND DATE(a.data_hora_consulta) BETWEEN ? AND ?";
        }
        
        if($id_clinica) {
            $where .= " AND a.id_clinica = ?";
        }
        
        $query = "SELECT 
                    COUNT(*) as total_agendamentos,
                    COUNT(CASE WHEN a.status_atendimento = 'Agendado' THEN 1 END) as agendados,
                    COUNT(CASE WHEN a.status_atendimento = 'Presente' THEN 1 END) as presentes,
                    COUNT(CASE WHEN a.status_atendimento = 'Faltou' THEN 1 END) as faltas,
                    COUNT(CASE WHEN a.status_atendimento = 'Cancelado' THEN 1 END) as cancelados
                  FROM " . $this->table_name . " a
                  " . $where;
        
        $stmt = $this->conn->prepare($query);
        
        $param_index = 1;
        if($data_inicio && $data_fim) {
            $stmt->bindParam($param_index++, $data_inicio);
            $stmt->bindParam($param_index++, $data_fim);
        }
        
        if($id_clinica) {
            $stmt->bindParam($param_index, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
?>
