<?php
require_once __DIR__ . '/../../config/database.php';

class Atendimento {
    private $conn;
    private $table_name = "atendimentos";
    
    public $id_atendimento;
    public $id_agendamento;
    public $id_clinica;
    public $id_paciente;
    public $id_profissional;
    public $data_atendimento;
    public $tipo_exame_realizado;
    public $valor_pago;
    public $forma_pagamento;
    public $observacao_atendimento;
    
    public function __construct() {
        $this->conn = getDBConnection();
    }
    
    /**
     * Criar novo atendimento
     */
    public function criar() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET id_agendamento=:id_agendamento, id_clinica=:id_clinica, id_paciente=:id_paciente,
                      id_profissional=:id_profissional, data_atendimento=:data_atendimento,
                      tipo_exame_realizado=:tipo_exame_realizado, valor_pago=:valor_pago,
                      forma_pagamento=:forma_pagamento, observacao_atendimento=:observacao_atendimento";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->id_agendamento = htmlspecialchars(strip_tags($this->id_agendamento));
        $this->id_clinica = htmlspecialchars(strip_tags($this->id_clinica));
        $this->id_paciente = htmlspecialchars(strip_tags($this->id_paciente));
        $this->id_profissional = htmlspecialchars(strip_tags($this->id_profissional));
        $this->data_atendimento = htmlspecialchars(strip_tags($this->data_atendimento));
        $this->tipo_exame_realizado = htmlspecialchars(strip_tags($this->tipo_exame_realizado));
        $this->valor_pago = floatval($this->valor_pago);
        $this->forma_pagamento = htmlspecialchars(strip_tags($this->forma_pagamento));
        $this->observacao_atendimento = htmlspecialchars(strip_tags($this->observacao_atendimento));
        
        // Bind dos valores
        $stmt->bindParam(":id_agendamento", $this->id_agendamento);
        $stmt->bindParam(":id_clinica", $this->id_clinica);
        $stmt->bindParam(":id_paciente", $this->id_paciente);
        $stmt->bindParam(":id_profissional", $this->id_profissional);
        $stmt->bindParam(":data_atendimento", $this->data_atendimento);
        $stmt->bindParam(":tipo_exame_realizado", $this->tipo_exame_realizado);
        $stmt->bindParam(":valor_pago", $this->valor_pago);
        $stmt->bindParam(":forma_pagamento", $this->forma_pagamento);
        $stmt->bindParam(":observacao_atendimento", $this->observacao_atendimento);
        
        if($stmt->execute()) {
            $this->id_atendimento = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    /**
     * Ler dados de um atendimento
     */
    public function lerUm() {
        $query = "SELECT at.*, p.nome_paciente, p.cpf, p.telefone_contato,
                         pr.nome_completo as nome_profissional, pr.tipo_registro,
                         c.nome_clinica, a.data_hora_consulta
                  FROM " . $this->table_name . " at
                  INNER JOIN agendamentos a ON at.id_agendamento = a.id_agendamento
                  INNER JOIN pacientes p ON at.id_paciente = p.id_paciente
                  INNER JOIN profissionais pr ON at.id_profissional = pr.id_profissional
                  INNER JOIN clinicas c ON at.id_clinica = c.id_clinica
                  WHERE at.id_atendimento = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_atendimento);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->id_agendamento = $row['id_agendamento'];
            $this->id_clinica = $row['id_clinica'];
            $this->id_paciente = $row['id_paciente'];
            $this->id_profissional = $row['id_profissional'];
            $this->data_atendimento = $row['data_atendimento'];
            $this->tipo_exame_realizado = $row['tipo_exame_realizado'];
            $this->valor_pago = $row['valor_pago'];
            $this->forma_pagamento = $row['forma_pagamento'];
            $this->observacao_atendimento = $row['observacao_atendimento'];
            return $row;
        }
        
        return false;
    }
    
    /**
     * Atualizar atendimento
     */
    public function atualizar() {
        $query = "UPDATE " . $this->table_name . " 
                  SET tipo_exame_realizado=:tipo_exame_realizado, valor_pago=:valor_pago,
                      forma_pagamento=:forma_pagamento, observacao_atendimento=:observacao_atendimento
                  WHERE id_atendimento=:id_atendimento";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->tipo_exame_realizado = htmlspecialchars(strip_tags($this->tipo_exame_realizado));
        $this->valor_pago = floatval($this->valor_pago);
        $this->forma_pagamento = htmlspecialchars(strip_tags($this->forma_pagamento));
        $this->observacao_atendimento = htmlspecialchars(strip_tags($this->observacao_atendimento));
        $this->id_atendimento = htmlspecialchars(strip_tags($this->id_atendimento));
        
        // Bind dos valores
        $stmt->bindParam(":tipo_exame_realizado", $this->tipo_exame_realizado);
        $stmt->bindParam(":valor_pago", $this->valor_pago);
        $stmt->bindParam(":forma_pagamento", $this->forma_pagamento);
        $stmt->bindParam(":observacao_atendimento", $this->observacao_atendimento);
        $stmt->bindParam(":id_atendimento", $this->id_atendimento);
        
        return $stmt->execute();
    }
    
    /**
     * Deletar atendimento
     */
    public function deletar() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id_atendimento = ?";
        
        $stmt = $this->conn->prepare($query);
        $this->id_atendimento = htmlspecialchars(strip_tags($this->id_atendimento));
        $stmt->bindParam(1, $this->id_atendimento);
        
        return $stmt->execute();
    }
    
    /**
     * Listar atendimentos por data
     */
    public function listarPorData($data, $id_clinica = null) {
        $where = "WHERE DATE(at.data_atendimento) = ?";
        
        if($id_clinica) {
            $where .= " AND at.id_clinica = ?";
        }
        
        $query = "SELECT at.*, p.nome_paciente, p.cpf, p.telefone_contato,
                         pr.nome_completo as nome_profissional, pr.tipo_registro,
                         c.nome_clinica
                  FROM " . $this->table_name . " at
                  INNER JOIN pacientes p ON at.id_paciente = p.id_paciente
                  INNER JOIN profissionais pr ON at.id_profissional = pr.id_profissional
                  INNER JOIN clinicas c ON at.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY at.data_atendimento ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Listar atendimentos por período
     */
    public function listarPorPeriodo($data_inicio, $data_fim, $id_clinica = null) {
        $where = "WHERE DATE(at.data_atendimento) BETWEEN ? AND ?";
        
        if($id_clinica) {
            $where .= " AND at.id_clinica = ?";
        }
        
        $query = "SELECT at.*, p.nome_paciente, p.cpf, p.telefone_contato,
                         pr.nome_completo as nome_profissional, pr.tipo_registro,
                         c.nome_clinica
                  FROM " . $this->table_name . " at
                  INNER JOIN pacientes p ON at.id_paciente = p.id_paciente
                  INNER JOIN profissionais pr ON at.id_profissional = pr.id_profissional
                  INNER JOIN clinicas c ON at.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY at.data_atendimento ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data_inicio);
        $stmt->bindParam(2, $data_fim);
        
        if($id_clinica) {
            $stmt->bindParam(3, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Obter resumo financeiro por data
     */
    public function obterResumoFinanceiro($data, $id_clinica = null) {
        $where = "WHERE DATE(at.data_atendimento) = ?";
        
        if($id_clinica) {
            $where .= " AND at.id_clinica = ?";
        }
        
        $query = "SELECT 
                    COUNT(*) as total_atendimentos,
                    SUM(at.valor_pago) as total_recebido,
                    at.forma_pagamento,
                    COUNT(*) as quantidade_forma_pagamento,
                    SUM(at.valor_pago) as valor_forma_pagamento
                  FROM " . $this->table_name . " at
                  " . $where . "
                  GROUP BY at.forma_pagamento
                  ORDER BY at.forma_pagamento";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter resumo por tipo de exame
     */
    public function obterResumoPorTipoExame($data, $id_clinica = null) {
        $where = "WHERE DATE(at.data_atendimento) = ?";
        
        if($id_clinica) {
            $where .= " AND at.id_clinica = ?";
        }
        
        $query = "SELECT 
                    at.tipo_exame_realizado,
                    COUNT(*) as quantidade,
                    SUM(at.valor_pago) as valor_total
                  FROM " . $this->table_name . " at
                  " . $where . "
                  GROUP BY at.tipo_exame_realizado
                  ORDER BY at.tipo_exame_realizado";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter resumo por profissional
     */
    public function obterResumoPorProfissional($data, $id_clinica = null) {
        $where = "WHERE DATE(at.data_atendimento) = ?";
        
        if($id_clinica) {
            $where .= " AND at.id_clinica = ?";
        }
        
        $query = "SELECT 
                    pr.nome_completo as nome_profissional,
                    pr.tipo_registro,
                    pr.salario,
                    COUNT(*) as quantidade_atendimentos,
                    SUM(at.valor_pago) as valor_total_atendimentos
                  FROM " . $this->table_name . " at
                  INNER JOIN profissionais pr ON at.id_profissional = pr.id_profissional
                  " . $where . "
                  GROUP BY at.id_profissional
                  ORDER BY pr.nome_completo";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Verificar se já existe atendimento para o agendamento
     */
    public function existeAtendimentoParaAgendamento($id_agendamento) {
        $query = "SELECT id_atendimento FROM " . $this->table_name . " WHERE id_agendamento = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id_agendamento);
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Obter estatísticas gerais
     */
    public function obterEstatisticas($data_inicio = null, $data_fim = null, $id_clinica = null) {
        $where = "WHERE 1=1";
        
        if($data_inicio && $data_fim) {
            $where .= " AND DATE(at.data_atendimento) BETWEEN ? AND ?";
        }
        
        if($id_clinica) {
            $where .= " AND at.id_clinica = ?";
        }
        
        $query = "SELECT 
                    COUNT(*) as total_atendimentos,
                    SUM(at.valor_pago) as total_faturamento,
                    AVG(at.valor_pago) as ticket_medio,
                    COUNT(DISTINCT at.id_paciente) as pacientes_unicos,
                    COUNT(DISTINCT at.id_profissional) as profissionais_ativos
                  FROM " . $this->table_name . " at
                  " . $where;
        
        $stmt = $this->conn->prepare($query);
        
        $param_index = 1;
        if($data_inicio && $data_fim) {
            $stmt->bindParam($param_index++, $data_inicio);
            $stmt->bindParam($param_index++, $data_fim);
        }
        
        if($id_clinica) {
            $stmt->bindParam($param_index, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter top pacientes por valor gasto
     */
    public function obterTopPacientes($limite = 10, $data_inicio = null, $data_fim = null, $id_clinica = null) {
        $where = "WHERE 1=1";
        
        if($data_inicio && $data_fim) {
            $where .= " AND DATE(at.data_atendimento) BETWEEN ? AND ?";
        }
        
        if($id_clinica) {
            $where .= " AND at.id_clinica = ?";
        }
        
        $query = "SELECT 
                    p.nome_paciente,
                    p.cpf,
                    p.telefone_contato,
                    COUNT(at.id_atendimento) as total_consultas,
                    SUM(at.valor_pago) as total_gasto
                  FROM " . $this->table_name . " at
                  INNER JOIN pacientes p ON at.id_paciente = p.id_paciente
                  " . $where . "
                  GROUP BY at.id_paciente
                  ORDER BY total_gasto DESC
                  LIMIT ?";
        
        $stmt = $this->conn->prepare($query);
        
        $param_index = 1;
        if($data_inicio && $data_fim) {
            $stmt->bindParam($param_index++, $data_inicio);
            $stmt->bindParam($param_index++, $data_fim);
        }
        
        if($id_clinica) {
            $stmt->bindParam($param_index++, $id_clinica);
        }
        
        $stmt->bindParam($param_index, $limite, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>
