<?php
require_once __DIR__ . '/../../config/database.php';

class Clinica {
    private $conn;
    private $table_name = "clinicas";
    
    public $id_clinica;
    public $nome_clinica;
    public $cnpj;
    public $endereco;
    public $telefone;
    public $email;
    public $data_cadastro;
    
    public function __construct() {
        $this->conn = getDBConnection();
    }
    
    /**
     * Criar nova clínica
     */
    public function criar() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET nome_clinica=:nome_clinica, cnpj=:cnpj, endereco=:endereco, 
                      telefone=:telefone, email=:email";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->nome_clinica = htmlspecialchars(strip_tags($this->nome_clinica));
        $this->cnpj = preg_replace('/[^0-9]/', '', $this->cnpj);
        $this->endereco = htmlspecialchars(strip_tags($this->endereco));
        $this->telefone = preg_replace('/[^0-9]/', '', $this->telefone);
        $this->email = htmlspecialchars(strip_tags($this->email));
        
        // Bind dos valores
        $stmt->bindParam(":nome_clinica", $this->nome_clinica);
        $stmt->bindParam(":cnpj", $this->cnpj);
        $stmt->bindParam(":endereco", $this->endereco);
        $stmt->bindParam(":telefone", $this->telefone);
        $stmt->bindParam(":email", $this->email);
        
        if($stmt->execute()) {
            $this->id_clinica = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    /**
     * Ler dados de uma clínica
     */
    public function lerUm() {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id_clinica = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_clinica);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->nome_clinica = $row['nome_clinica'];
            $this->cnpj = $row['cnpj'];
            $this->endereco = $row['endereco'];
            $this->telefone = $row['telefone'];
            $this->email = $row['email'];
            $this->data_cadastro = $row['data_cadastro'];
            return true;
        }
        
        return false;
    }
    
    /**
     * Atualizar clínica
     */
    public function atualizar() {
        $query = "UPDATE " . $this->table_name . " 
                  SET nome_clinica=:nome_clinica, cnpj=:cnpj, endereco=:endereco, 
                      telefone=:telefone, email=:email 
                  WHERE id_clinica=:id_clinica";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->nome_clinica = htmlspecialchars(strip_tags($this->nome_clinica));
        $this->cnpj = preg_replace('/[^0-9]/', '', $this->cnpj);
        $this->endereco = htmlspecialchars(strip_tags($this->endereco));
        $this->telefone = preg_replace('/[^0-9]/', '', $this->telefone);
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->id_clinica = htmlspecialchars(strip_tags($this->id_clinica));
        
        // Bind dos valores
        $stmt->bindParam(":nome_clinica", $this->nome_clinica);
        $stmt->bindParam(":cnpj", $this->cnpj);
        $stmt->bindParam(":endereco", $this->endereco);
        $stmt->bindParam(":telefone", $this->telefone);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":id_clinica", $this->id_clinica);
        
        return $stmt->execute();
    }
    
    /**
     * Deletar clínica
     */
    public function deletar() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id_clinica = ?";
        
        $stmt = $this->conn->prepare($query);
        $this->id_clinica = htmlspecialchars(strip_tags($this->id_clinica));
        $stmt->bindParam(1, $this->id_clinica);
        
        return $stmt->execute();
    }
    
    /**
     * Listar todas as clínicas
     */
    public function listarTodas() {
        $query = "SELECT * FROM " . $this->table_name . " ORDER BY nome_clinica ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Verificar se CNPJ já existe
     */
    public function cnpjExiste($cnpj, $id_clinica = null) {
        $query = "SELECT id_clinica FROM " . $this->table_name . " WHERE cnpj = ?";
        
        if($id_clinica) {
            $query .= " AND id_clinica != ?";
        }
        
        $stmt = $this->conn->prepare($query);
        $cnpj = preg_replace('/[^0-9]/', '', $cnpj);
        $stmt->bindParam(1, $cnpj);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Buscar clínicas por termo
     */
    public function buscar($termo) {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE nome_clinica LIKE ? OR cnpj LIKE ? OR endereco LIKE ? 
                  ORDER BY nome_clinica ASC";
        
        $stmt = $this->conn->prepare($query);
        $termo = "%{$termo}%";
        $stmt->bindParam(1, $termo);
        $stmt->bindParam(2, $termo);
        $stmt->bindParam(3, $termo);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Contar total de clínicas
     */
    public function contarTotal() {
        $query = "SELECT COUNT(*) as total FROM " . $this->table_name;
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['total'];
    }
}
?>
