<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/config.php';

class DocumentoClinica {
    private $conn;
    private $table_name = "documentos_clinica";
    
    public $id_documento;
    public $id_clinica;
    public $tipo_documento;
    public $nome_arquivo;
    public $caminho_arquivo;
    public $data_upload;
    public $data_validade;
    public $tamanho_mb;
    
    public function __construct() {
        $this->conn = getDBConnection();
    }
    
    /**
     * Criar novo documento
     */
    public function criar() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET id_clinica=:id_clinica, tipo_documento=:tipo_documento, 
                      nome_arquivo=:nome_arquivo, caminho_arquivo=:caminho_arquivo, 
                      data_validade=:data_validade, tamanho_mb=:tamanho_mb";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->id_clinica = htmlspecialchars(strip_tags($this->id_clinica));
        $this->tipo_documento = htmlspecialchars(strip_tags($this->tipo_documento));
        $this->nome_arquivo = htmlspecialchars(strip_tags($this->nome_arquivo));
        $this->caminho_arquivo = htmlspecialchars(strip_tags($this->caminho_arquivo));
        $this->data_validade = $this->data_validade ?: null;
        $this->tamanho_mb = floatval($this->tamanho_mb);
        
        // Bind dos valores
        $stmt->bindParam(":id_clinica", $this->id_clinica);
        $stmt->bindParam(":tipo_documento", $this->tipo_documento);
        $stmt->bindParam(":nome_arquivo", $this->nome_arquivo);
        $stmt->bindParam(":caminho_arquivo", $this->caminho_arquivo);
        $stmt->bindParam(":data_validade", $this->data_validade);
        $stmt->bindParam(":tamanho_mb", $this->tamanho_mb);
        
        if($stmt->execute()) {
            $this->id_documento = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    /**
     * Ler documento por ID
     */
    public function lerUm() {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id_documento = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_documento);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->id_clinica = $row['id_clinica'];
            $this->tipo_documento = $row['tipo_documento'];
            $this->nome_arquivo = $row['nome_arquivo'];
            $this->caminho_arquivo = $row['caminho_arquivo'];
            $this->data_upload = $row['data_upload'];
            $this->data_validade = $row['data_validade'];
            $this->tamanho_mb = $row['tamanho_mb'];
            return true;
        }
        
        return false;
    }
    
    /**
     * Atualizar documento
     */
    public function atualizar() {
        $query = "UPDATE " . $this->table_name . " 
                  SET tipo_documento=:tipo_documento, nome_arquivo=:nome_arquivo, 
                      caminho_arquivo=:caminho_arquivo, data_validade=:data_validade, 
                      tamanho_mb=:tamanho_mb 
                  WHERE id_documento=:id_documento";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->tipo_documento = htmlspecialchars(strip_tags($this->tipo_documento));
        $this->nome_arquivo = htmlspecialchars(strip_tags($this->nome_arquivo));
        $this->caminho_arquivo = htmlspecialchars(strip_tags($this->caminho_arquivo));
        $this->data_validade = $this->data_validade ?: null;
        $this->tamanho_mb = floatval($this->tamanho_mb);
        $this->id_documento = htmlspecialchars(strip_tags($this->id_documento));
        
        // Bind dos valores
        $stmt->bindParam(":tipo_documento", $this->tipo_documento);
        $stmt->bindParam(":nome_arquivo", $this->nome_arquivo);
        $stmt->bindParam(":caminho_arquivo", $this->caminho_arquivo);
        $stmt->bindParam(":data_validade", $this->data_validade);
        $stmt->bindParam(":tamanho_mb", $this->tamanho_mb);
        $stmt->bindParam(":id_documento", $this->id_documento);
        
        return $stmt->execute();
    }
    
    /**
     * Deletar documento
     */
    public function deletar() {
        // Primeiro, deletar o arquivo físico
        if(file_exists($this->caminho_arquivo)) {
            unlink($this->caminho_arquivo);
        }
        
        $query = "DELETE FROM " . $this->table_name . " WHERE id_documento = ?";
        
        $stmt = $this->conn->prepare($query);
        $this->id_documento = htmlspecialchars(strip_tags($this->id_documento));
        $stmt->bindParam(1, $this->id_documento);
        
        return $stmt->execute();
    }
    
    /**
     * Listar documentos de uma clínica
     */
    public function listarPorClinica($id_clinica) {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE id_clinica = ? 
                  ORDER BY tipo_documento ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id_clinica);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Verificar se documento já existe para a clínica
     */
    public function documentoExiste($id_clinica, $tipo_documento, $id_documento = null) {
        $query = "SELECT id_documento FROM " . $this->table_name . " 
                  WHERE id_clinica = ? AND tipo_documento = ?";
        
        if($id_documento) {
            $query .= " AND id_documento != ?";
        }
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id_clinica);
        $stmt->bindParam(2, $tipo_documento);
        
        if($id_documento) {
            $stmt->bindParam(3, $id_documento);
        }
        
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Fazer upload do arquivo
     */
    public function fazerUpload($arquivo, $id_clinica, $tipo_documento) {
        // Validar arquivo
        $erros = validarUpload($arquivo);
        if(!empty($erros)) {
            return ['sucesso' => false, 'erros' => $erros];
        }
        
        // Criar diretório se não existir
        $diretorio = UPLOAD_PATH_CLINICAS . $id_clinica . '/';
        if(!is_dir($diretorio)) {
            mkdir($diretorio, 0755, true);
        }
        
        // Gerar nome único para o arquivo
        $extensao = pathinfo($arquivo['name'], PATHINFO_EXTENSION);
        $nome_arquivo = $tipo_documento . '_' . time() . '.' . $extensao;
        $caminho_completo = $diretorio . $nome_arquivo;
        
        // Mover arquivo
        if(move_uploaded_file($arquivo['tmp_name'], $caminho_completo)) {
            // Calcular tamanho em MB
            $tamanho_mb = round($arquivo['size'] / (1024 * 1024), 2);
            
            // Definir propriedades
            $this->id_clinica = $id_clinica;
            $this->tipo_documento = $tipo_documento;
            $this->nome_arquivo = $arquivo['name'];
            $this->caminho_arquivo = $caminho_completo;
            $this->tamanho_mb = $tamanho_mb;
            
            // Salvar no banco
            if($this->criar()) {
                return ['sucesso' => true, 'id_documento' => $this->id_documento];
            } else {
                // Se falhou ao salvar no banco, deletar arquivo
                unlink($caminho_completo);
                return ['sucesso' => false, 'erros' => ['Erro ao salvar no banco de dados']];
            }
        } else {
            return ['sucesso' => false, 'erros' => ['Erro ao fazer upload do arquivo']];
        }
    }
    
    /**
     * Listar documentos vencidos ou próximos do vencimento
     */
    public function listarVencimentos($dias_antecedencia = 30) {
        $query = "SELECT d.*, c.nome_clinica 
                  FROM " . $this->table_name . " d
                  INNER JOIN clinicas c ON d.id_clinica = c.id_clinica
                  WHERE d.data_validade IS NOT NULL 
                  AND d.data_validade <= DATE_ADD(CURDATE(), INTERVAL ? DAY)
                  ORDER BY d.data_validade ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $dias_antecedencia);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Obter estatísticas de documentos
     */
    public function obterEstatisticas($id_clinica = null) {
        $where = $id_clinica ? "WHERE id_clinica = ?" : "";
        
        $query = "SELECT 
                    COUNT(*) as total_documentos,
                    COUNT(CASE WHEN data_validade IS NOT NULL AND data_validade < CURDATE() THEN 1 END) as vencidos,
                    COUNT(CASE WHEN data_validade IS NOT NULL AND data_validade BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY) THEN 1 END) as vencendo
                  FROM " . $this->table_name . " " . $where;
        
        $stmt = $this->conn->prepare($query);
        
        if($id_clinica) {
            $stmt->bindParam(1, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
?>
