<?php
require_once __DIR__ . '/../../config/database.php';

class DespesaDiaria {
    private $conn;
    private $table_name = "despesas_diarias";
    
    public $id_despesa;
    public $id_clinica;
    public $data_despesa;
    public $descricao;
    public $valor;
    public $data_registro;
    
    public function __construct() {
        $this->conn = getDBConnection();
    }
    
    /**
     * Criar nova despesa
     */
    public function criar() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET id_clinica=:id_clinica, data_despesa=:data_despesa, 
                      descricao=:descricao, valor=:valor";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->id_clinica = htmlspecialchars(strip_tags($this->id_clinica));
        $this->data_despesa = htmlspecialchars(strip_tags($this->data_despesa));
        $this->descricao = htmlspecialchars(strip_tags($this->descricao));
        $this->valor = floatval($this->valor);
        
        // Bind dos valores
        $stmt->bindParam(":id_clinica", $this->id_clinica);
        $stmt->bindParam(":data_despesa", $this->data_despesa);
        $stmt->bindParam(":descricao", $this->descricao);
        $stmt->bindParam(":valor", $this->valor);
        
        if($stmt->execute()) {
            $this->id_despesa = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    /**
     * Ler dados de uma despesa
     */
    public function lerUm() {
        $query = "SELECT d.*, c.nome_clinica 
                  FROM " . $this->table_name . " d
                  INNER JOIN clinicas c ON d.id_clinica = c.id_clinica
                  WHERE d.id_despesa = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_despesa);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->id_clinica = $row['id_clinica'];
            $this->data_despesa = $row['data_despesa'];
            $this->descricao = $row['descricao'];
            $this->valor = $row['valor'];
            $this->data_registro = $row['data_registro'];
            return $row;
        }
        
        return false;
    }
    
    /**
     * Atualizar despesa
     */
    public function atualizar() {
        $query = "UPDATE " . $this->table_name . " 
                  SET data_despesa=:data_despesa, descricao=:descricao, valor=:valor
                  WHERE id_despesa=:id_despesa";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->data_despesa = htmlspecialchars(strip_tags($this->data_despesa));
        $this->descricao = htmlspecialchars(strip_tags($this->descricao));
        $this->valor = floatval($this->valor);
        $this->id_despesa = htmlspecialchars(strip_tags($this->id_despesa));
        
        // Bind dos valores
        $stmt->bindParam(":data_despesa", $this->data_despesa);
        $stmt->bindParam(":descricao", $this->descricao);
        $stmt->bindParam(":valor", $this->valor);
        $stmt->bindParam(":id_despesa", $this->id_despesa);
        
        return $stmt->execute();
    }
    
    /**
     * Deletar despesa
     */
    public function deletar() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id_despesa = ?";
        
        $stmt = $this->conn->prepare($query);
        $this->id_despesa = htmlspecialchars(strip_tags($this->id_despesa));
        $stmt->bindParam(1, $this->id_despesa);
        
        return $stmt->execute();
    }
    
    /**
     * Listar despesas por data
     */
    public function listarPorData($data, $id_clinica = null) {
        $where = "WHERE d.data_despesa = ?";
        
        if($id_clinica) {
            $where .= " AND d.id_clinica = ?";
        }
        
        $query = "SELECT d.*, c.nome_clinica 
                  FROM " . $this->table_name . " d
                  INNER JOIN clinicas c ON d.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY d.data_registro DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Listar despesas por período
     */
    public function listarPorPeriodo($data_inicio, $data_fim, $id_clinica = null) {
        $where = "WHERE d.data_despesa BETWEEN ? AND ?";
        
        if($id_clinica) {
            $where .= " AND d.id_clinica = ?";
        }
        
        $query = "SELECT d.*, c.nome_clinica 
                  FROM " . $this->table_name . " d
                  INNER JOIN clinicas c ON d.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY d.data_despesa DESC, d.data_registro DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data_inicio);
        $stmt->bindParam(2, $data_fim);
        
        if($id_clinica) {
            $stmt->bindParam(3, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Obter total de despesas por data
     */
    public function obterTotalPorData($data, $id_clinica = null) {
        $where = "WHERE data_despesa = ?";
        
        if($id_clinica) {
            $where .= " AND id_clinica = ?";
        }
        
        $query = "SELECT 
                    COUNT(*) as total_despesas,
                    SUM(valor) as valor_total
                  FROM " . $this->table_name . " 
                  " . $where;
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter total de despesas por período
     */
    public function obterTotalPorPeriodo($data_inicio, $data_fim, $id_clinica = null) {
        $where = "WHERE data_despesa BETWEEN ? AND ?";
        
        if($id_clinica) {
            $where .= " AND id_clinica = ?";
        }
        
        $query = "SELECT 
                    COUNT(*) as total_despesas,
                    SUM(valor) as valor_total,
                    AVG(valor) as valor_medio
                  FROM " . $this->table_name . " 
                  " . $where;
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data_inicio);
        $stmt->bindParam(2, $data_fim);
        
        if($id_clinica) {
            $stmt->bindParam(3, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter despesas agrupadas por descrição
     */
    public function obterDespesasAgrupadas($data_inicio, $data_fim, $id_clinica = null) {
        $where = "WHERE data_despesa BETWEEN ? AND ?";
        
        if($id_clinica) {
            $where .= " AND id_clinica = ?";
        }
        
        $query = "SELECT 
                    descricao,
                    COUNT(*) as quantidade,
                    SUM(valor) as valor_total,
                    AVG(valor) as valor_medio
                  FROM " . $this->table_name . " 
                  " . $where . "
                  GROUP BY descricao
                  ORDER BY valor_total DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data_inicio);
        $stmt->bindParam(2, $data_fim);
        
        if($id_clinica) {
            $stmt->bindParam(3, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter maiores despesas
     */
    public function obterMaioresDespesas($limite = 10, $data_inicio = null, $data_fim = null, $id_clinica = null) {
        $where = "WHERE 1=1";
        
        if($data_inicio && $data_fim) {
            $where .= " AND d.data_despesa BETWEEN ? AND ?";
        }
        
        if($id_clinica) {
            $where .= " AND d.id_clinica = ?";
        }
        
        $query = "SELECT d.*, c.nome_clinica 
                  FROM " . $this->table_name . " d
                  INNER JOIN clinicas c ON d.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY d.valor DESC
                  LIMIT ?";
        
        $stmt = $this->conn->prepare($query);
        
        $param_index = 1;
        if($data_inicio && $data_fim) {
            $stmt->bindParam($param_index++, $data_inicio);
            $stmt->bindParam($param_index++, $data_fim);
        }
        
        if($id_clinica) {
            $stmt->bindParam($param_index++, $id_clinica);
        }
        
        $stmt->bindParam($param_index, $limite, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter evolução mensal das despesas
     */
    public function obterEvolucaoMensal($ano, $id_clinica = null) {
        $where = "WHERE YEAR(data_despesa) = ?";
        
        if($id_clinica) {
            $where .= " AND id_clinica = ?";
        }
        
        $query = "SELECT 
                    MONTH(data_despesa) as mes,
                    COUNT(*) as quantidade,
                    SUM(valor) as valor_total
                  FROM " . $this->table_name . " 
                  " . $where . "
                  GROUP BY MONTH(data_despesa)
                  ORDER BY mes";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $ano);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>
