<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../atendimentos/Atendimento.php';
require_once __DIR__ . '/DespesaDiaria.php';

class FechamentoDiario {
    private $conn;
    private $table_name = "fechamento_diario";
    
    public $id_fechamento;
    public $id_clinica;
    public $data_fechamento;
    public $total_recebido;
    public $total_despesas;
    public $saldo_dia;
    public $detalhes_pagamento;
    public $detalhes_exame;
    public $detalhes_profissionais;
    
    public function __construct() {
        $this->conn = getDBConnection();
    }
    
    /**
     * Processar fechamento do dia
     */
    public function processarFechamento($data, $id_clinica) {
        try {
            $this->conn->beginTransaction();
            
            // Instanciar classes necessárias
            $atendimento = new Atendimento();
            $despesa = new DespesaDiaria();
            
            // Obter resumo financeiro dos atendimentos
            $resumo_pagamentos = $atendimento->obterResumoFinanceiro($data, $id_clinica);
            $resumo_exames = $atendimento->obterResumoPorTipoExame($data, $id_clinica);
            $resumo_profissionais = $atendimento->obterResumoPorProfissional($data, $id_clinica);
            
            // Obter total de despesas
            $total_despesas_data = $despesa->obterTotalPorData($data, $id_clinica);
            
            // Calcular totais
            $total_recebido = 0;
            $detalhes_pagamento = [];
            
            foreach ($resumo_pagamentos as $pagamento) {
                $total_recebido += $pagamento['valor_forma_pagamento'];
                $detalhes_pagamento[$pagamento['forma_pagamento']] = [
                    'quantidade' => $pagamento['quantidade_forma_pagamento'],
                    'valor' => $pagamento['valor_forma_pagamento']
                ];
            }
            
            // Organizar detalhes dos exames
            $detalhes_exame = [];
            foreach ($resumo_exames as $exame) {
                $detalhes_exame[$exame['tipo_exame_realizado']] = [
                    'quantidade' => $exame['quantidade'],
                    'valor' => $exame['valor_total']
                ];
            }
            
            // Organizar detalhes dos profissionais
            $detalhes_profissionais = [];
            foreach ($resumo_profissionais as $prof) {
                $detalhes_profissionais[$prof['nome_profissional']] = [
                    'tipo_registro' => $prof['tipo_registro'],
                    'salario' => $prof['salario'],
                    'quantidade_atendimentos' => $prof['quantidade_atendimentos'],
                    'valor_total_atendimentos' => $prof['valor_total_atendimentos']
                ];
            }
            
            $total_despesas = $total_despesas_data['valor_total'] ?? 0;
            $saldo_dia = $total_recebido - $total_despesas;
            
            // Verificar se já existe fechamento para esta data
            if ($this->existeFechamento($data, $id_clinica)) {
                // Atualizar fechamento existente
                $this->atualizarFechamento($data, $id_clinica, $total_recebido, $total_despesas, $saldo_dia, 
                                         $detalhes_pagamento, $detalhes_exame, $detalhes_profissionais);
            } else {
                // Criar novo fechamento
                $this->criarFechamento($data, $id_clinica, $total_recebido, $total_despesas, $saldo_dia, 
                                     $detalhes_pagamento, $detalhes_exame, $detalhes_profissionais);
            }
            
            $this->conn->commit();
            return true;
            
        } catch (Exception $e) {
            $this->conn->rollback();
            throw $e;
        }
    }
    
    /**
     * Criar novo fechamento
     */
    private function criarFechamento($data, $id_clinica, $total_recebido, $total_despesas, $saldo_dia, 
                                   $detalhes_pagamento, $detalhes_exame, $detalhes_profissionais) {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET id_clinica=:id_clinica, data_fechamento=:data_fechamento, 
                      total_recebido=:total_recebido, total_despesas=:total_despesas, 
                      saldo_dia=:saldo_dia, detalhes_pagamento=:detalhes_pagamento,
                      detalhes_exame=:detalhes_exame, detalhes_profissionais=:detalhes_profissionais";
        
        $stmt = $this->conn->prepare($query);
        
        $stmt->bindParam(":id_clinica", $id_clinica);
        $stmt->bindParam(":data_fechamento", $data);
        $stmt->bindParam(":total_recebido", $total_recebido);
        $stmt->bindParam(":total_despesas", $total_despesas);
        $stmt->bindParam(":saldo_dia", $saldo_dia);
        $stmt->bindParam(":detalhes_pagamento", json_encode($detalhes_pagamento));
        $stmt->bindParam(":detalhes_exame", json_encode($detalhes_exame));
        $stmt->bindParam(":detalhes_profissionais", json_encode($detalhes_profissionais));
        
        if($stmt->execute()) {
            $this->id_fechamento = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    /**
     * Atualizar fechamento existente
     */
    private function atualizarFechamento($data, $id_clinica, $total_recebido, $total_despesas, $saldo_dia, 
                                       $detalhes_pagamento, $detalhes_exame, $detalhes_profissionais) {
        $query = "UPDATE " . $this->table_name . " 
                  SET total_recebido=:total_recebido, total_despesas=:total_despesas, 
                      saldo_dia=:saldo_dia, detalhes_pagamento=:detalhes_pagamento,
                      detalhes_exame=:detalhes_exame, detalhes_profissionais=:detalhes_profissionais
                  WHERE id_clinica=:id_clinica AND data_fechamento=:data_fechamento";
        
        $stmt = $this->conn->prepare($query);
        
        $stmt->bindParam(":total_recebido", $total_recebido);
        $stmt->bindParam(":total_despesas", $total_despesas);
        $stmt->bindParam(":saldo_dia", $saldo_dia);
        $stmt->bindParam(":detalhes_pagamento", json_encode($detalhes_pagamento));
        $stmt->bindParam(":detalhes_exame", json_encode($detalhes_exame));
        $stmt->bindParam(":detalhes_profissionais", json_encode($detalhes_profissionais));
        $stmt->bindParam(":id_clinica", $id_clinica);
        $stmt->bindParam(":data_fechamento", $data);
        
        return $stmt->execute();
    }
    
    /**
     * Verificar se existe fechamento para a data
     */
    public function existeFechamento($data, $id_clinica) {
        $query = "SELECT id_fechamento FROM " . $this->table_name . " 
                  WHERE data_fechamento = ? AND id_clinica = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data);
        $stmt->bindParam(2, $id_clinica);
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Obter fechamento por data
     */
    public function obterFechamentoPorData($data, $id_clinica) {
        $query = "SELECT f.*, c.nome_clinica 
                  FROM " . $this->table_name . " f
                  INNER JOIN clinicas c ON f.id_clinica = c.id_clinica
                  WHERE f.data_fechamento = ? AND f.id_clinica = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data);
        $stmt->bindParam(2, $id_clinica);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            // Decodificar JSONs
            $row['detalhes_pagamento'] = json_decode($row['detalhes_pagamento'], true);
            $row['detalhes_exame'] = json_decode($row['detalhes_exame'], true);
            $row['detalhes_profissionais'] = json_decode($row['detalhes_profissionais'], true);
        }
        
        return $row;
    }
    
    /**
     * Listar fechamentos por período
     */
    public function listarPorPeriodo($data_inicio, $data_fim, $id_clinica = null) {
        $where = "WHERE f.data_fechamento BETWEEN ? AND ?";
        
        if($id_clinica) {
            $where .= " AND f.id_clinica = ?";
        }
        
        $query = "SELECT f.*, c.nome_clinica 
                  FROM " . $this->table_name . " f
                  INNER JOIN clinicas c ON f.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY f.data_fechamento DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data_inicio);
        $stmt->bindParam(2, $data_fim);
        
        if($id_clinica) {
            $stmt->bindParam(3, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Obter resumo do período
     */
    public function obterResumoPeriodo($data_inicio, $data_fim, $id_clinica = null) {
        $where = "WHERE data_fechamento BETWEEN ? AND ?";
        
        if($id_clinica) {
            $where .= " AND id_clinica = ?";
        }
        
        $query = "SELECT 
                    COUNT(*) as dias_fechados,
                    SUM(total_recebido) as total_recebido_periodo,
                    SUM(total_despesas) as total_despesas_periodo,
                    SUM(saldo_dia) as saldo_periodo,
                    AVG(total_recebido) as media_recebido_dia,
                    AVG(total_despesas) as media_despesas_dia,
                    AVG(saldo_dia) as media_saldo_dia
                  FROM " . $this->table_name . " 
                  " . $where;
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $data_inicio);
        $stmt->bindParam(2, $data_fim);
        
        if($id_clinica) {
            $stmt->bindParam(3, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter evolução mensal
     */
    public function obterEvolucaoMensal($ano, $id_clinica = null) {
        $where = "WHERE YEAR(data_fechamento) = ?";
        
        if($id_clinica) {
            $where .= " AND id_clinica = ?";
        }
        
        $query = "SELECT 
                    MONTH(data_fechamento) as mes,
                    COUNT(*) as dias_fechados,
                    SUM(total_recebido) as total_recebido,
                    SUM(total_despesas) as total_despesas,
                    SUM(saldo_dia) as saldo_mes
                  FROM " . $this->table_name . " 
                  " . $where . "
                  GROUP BY MONTH(data_fechamento)
                  ORDER BY mes";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $ano);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter melhores dias
     */
    public function obterMelhoresDias($limite = 10, $data_inicio = null, $data_fim = null, $id_clinica = null) {
        $where = "WHERE 1=1";
        
        if($data_inicio && $data_fim) {
            $where .= " AND f.data_fechamento BETWEEN ? AND ?";
        }
        
        if($id_clinica) {
            $where .= " AND f.id_clinica = ?";
        }
        
        $query = "SELECT f.*, c.nome_clinica 
                  FROM " . $this->table_name . " f
                  INNER JOIN clinicas c ON f.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY f.saldo_dia DESC
                  LIMIT ?";
        
        $stmt = $this->conn->prepare($query);
        
        $param_index = 1;
        if($data_inicio && $data_fim) {
            $stmt->bindParam($param_index++, $data_inicio);
            $stmt->bindParam($param_index++, $data_fim);
        }
        
        if($id_clinica) {
            $stmt->bindParam($param_index++, $id_clinica);
        }
        
        $stmt->bindParam($param_index, $limite, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Deletar fechamento
     */
    public function deletar() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id_fechamento = ?";
        
        $stmt = $this->conn->prepare($query);
        $this->id_fechamento = htmlspecialchars(strip_tags($this->id_fechamento));
        $stmt->bindParam(1, $this->id_fechamento);
        
        return $stmt->execute();
    }
}
?>
