<?php
require_once __DIR__ . '/../../config/database.php';

class Paciente {
    private $conn;
    private $table_name = "pacientes";
    
    public $id_paciente;
    public $nome_paciente;
    public $cpf;
    public $renach;
    public $telefone_contato;
    public $data_cadastro;
    
    public function __construct() {
        $this->conn = getDBConnection();
    }
    
    /**
     * Criar novo paciente
     */
    public function criar() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET nome_paciente=:nome_paciente, cpf=:cpf, renach=:renach, 
                      telefone_contato=:telefone_contato";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->nome_paciente = htmlspecialchars(strip_tags($this->nome_paciente));
        $this->cpf = preg_replace('/[^0-9]/', '', $this->cpf);
        $this->renach = htmlspecialchars(strip_tags($this->renach));
        $this->telefone_contato = preg_replace('/[^0-9]/', '', $this->telefone_contato);
        
        // Bind dos valores
        $stmt->bindParam(":nome_paciente", $this->nome_paciente);
        $stmt->bindParam(":cpf", $this->cpf);
        $stmt->bindParam(":renach", $this->renach);
        $stmt->bindParam(":telefone_contato", $this->telefone_contato);
        
        if($stmt->execute()) {
            $this->id_paciente = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    /**
     * Ler dados de um paciente
     */
    public function lerUm() {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id_paciente = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_paciente);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->nome_paciente = $row['nome_paciente'];
            $this->cpf = $row['cpf'];
            $this->renach = $row['renach'];
            $this->telefone_contato = $row['telefone_contato'];
            $this->data_cadastro = $row['data_cadastro'];
            return true;
        }
        
        return false;
    }
    
    /**
     * Atualizar paciente
     */
    public function atualizar() {
        $query = "UPDATE " . $this->table_name . " 
                  SET nome_paciente=:nome_paciente, cpf=:cpf, renach=:renach, 
                      telefone_contato=:telefone_contato 
                  WHERE id_paciente=:id_paciente";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->nome_paciente = htmlspecialchars(strip_tags($this->nome_paciente));
        $this->cpf = preg_replace('/[^0-9]/', '', $this->cpf);
        $this->renach = htmlspecialchars(strip_tags($this->renach));
        $this->telefone_contato = preg_replace('/[^0-9]/', '', $this->telefone_contato);
        $this->id_paciente = htmlspecialchars(strip_tags($this->id_paciente));
        
        // Bind dos valores
        $stmt->bindParam(":nome_paciente", $this->nome_paciente);
        $stmt->bindParam(":cpf", $this->cpf);
        $stmt->bindParam(":renach", $this->renach);
        $stmt->bindParam(":telefone_contato", $this->telefone_contato);
        $stmt->bindParam(":id_paciente", $this->id_paciente);
        
        return $stmt->execute();
    }
    
    /**
     * Deletar paciente
     */
    public function deletar() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id_paciente = ?";
        
        $stmt = $this->conn->prepare($query);
        $this->id_paciente = htmlspecialchars(strip_tags($this->id_paciente));
        $stmt->bindParam(1, $this->id_paciente);
        
        return $stmt->execute();
    }
    
    /**
     * Listar todos os pacientes
     */
    public function listarTodos() {
        $query = "SELECT * FROM " . $this->table_name . " ORDER BY nome_paciente ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Verificar se CPF já existe
     */
    public function cpfExiste($cpf, $id_paciente = null) {
        $query = "SELECT id_paciente FROM " . $this->table_name . " WHERE cpf = ?";
        
        if($id_paciente) {
            $query .= " AND id_paciente != ?";
        }
        
        $stmt = $this->conn->prepare($query);
        $cpf = preg_replace('/[^0-9]/', '', $cpf);
        $stmt->bindParam(1, $cpf);
        
        if($id_paciente) {
            $stmt->bindParam(2, $id_paciente);
        }
        
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Buscar pacientes por termo
     */
    public function buscar($termo) {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE nome_paciente LIKE ? OR cpf LIKE ? OR renach LIKE ? OR telefone_contato LIKE ?
                  ORDER BY nome_paciente ASC";
        
        $stmt = $this->conn->prepare($query);
        $termo = "%{$termo}%";
        $stmt->bindParam(1, $termo);
        $stmt->bindParam(2, $termo);
        $stmt->bindParam(3, $termo);
        $stmt->bindParam(4, $termo);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Buscar paciente por CPF
     */
    public function buscarPorCPF($cpf) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE cpf = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $cpf = preg_replace('/[^0-9]/', '', $cpf);
        $stmt->bindParam(1, $cpf);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->id_paciente = $row['id_paciente'];
            $this->nome_paciente = $row['nome_paciente'];
            $this->cpf = $row['cpf'];
            $this->renach = $row['renach'];
            $this->telefone_contato = $row['telefone_contato'];
            $this->data_cadastro = $row['data_cadastro'];
            return true;
        }
        
        return false;
    }
    
    /**
     * Criar ou atualizar paciente (usado no agendamento)
     */
    public function criarOuAtualizar() {
        // Primeiro, verificar se o paciente já existe pelo CPF
        if($this->buscarPorCPF($this->cpf)) {
            // Se existe, atualizar os dados
            return $this->atualizar();
        } else {
            // Se não existe, criar novo
            return $this->criar();
        }
    }
    
    /**
     * Obter histórico de consultas do paciente
     */
    public function obterHistoricoConsultas() {
        $query = "SELECT a.*, c.nome_clinica, p.nome_completo as nome_profissional,
                         at.tipo_exame_realizado, at.valor_pago, at.forma_pagamento, at.data_atendimento
                  FROM agendamentos a
                  INNER JOIN clinicas c ON a.id_clinica = c.id_clinica
                  LEFT JOIN profissionais p ON a.id_profissional = p.id_profissional
                  LEFT JOIN atendimentos at ON a.id_agendamento = at.id_agendamento
                  WHERE a.id_paciente = ?
                  ORDER BY a.data_hora_consulta DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_paciente);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Contar total de pacientes
     */
    public function contarTotal() {
        $query = "SELECT COUNT(*) as total FROM " . $this->table_name;
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['total'];
    }
    
    /**
     * Obter estatísticas do paciente
     */
    public function obterEstatisticas() {
        $query = "SELECT 
                    COUNT(a.id_agendamento) as total_agendamentos,
                    COUNT(CASE WHEN a.status_atendimento = 'Presente' THEN 1 END) as total_presencas,
                    COUNT(CASE WHEN a.status_atendimento = 'Faltou' THEN 1 END) as total_faltas,
                    COALESCE(SUM(at.valor_pago), 0) as total_gasto
                  FROM agendamentos a
                  LEFT JOIN atendimentos at ON a.id_agendamento = at.id_agendamento
                  WHERE a.id_paciente = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_paciente);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Listar pacientes com mais consultas
     */
    public function listarMaisFrequentes($limite = 10) {
        $query = "SELECT p.*, COUNT(a.id_agendamento) as total_consultas
                  FROM " . $this->table_name . " p
                  LEFT JOIN agendamentos a ON p.id_paciente = a.id_paciente
                  GROUP BY p.id_paciente
                  ORDER BY total_consultas DESC, p.nome_paciente ASC
                  LIMIT ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $limite, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt;
    }
}
?>
