<?php
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/config.php';

class Profissional {
    private $conn;
    private $table_name = "profissionais";
    
    public $id_profissional;
    public $nome_completo;
    public $cpf;
    public $registro_conselho;
    public $tipo_registro;
    public $salario;
    public $data_contratacao;
    public $nome_arquivo_documento;
    public $caminho_arquivo_documento;
    public $tamanho_mb_documento;
    public $data_cadastro;
    
    public function __construct() {
        $this->conn = getDBConnection();
    }
    
    /**
     * Criar novo profissional
     */
    public function criar() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET nome_completo=:nome_completo, cpf=:cpf, registro_conselho=:registro_conselho, 
                      tipo_registro=:tipo_registro, salario=:salario, data_contratacao=:data_contratacao,
                      nome_arquivo_documento=:nome_arquivo_documento, caminho_arquivo_documento=:caminho_arquivo_documento,
                      tamanho_mb_documento=:tamanho_mb_documento";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->nome_completo = htmlspecialchars(strip_tags($this->nome_completo));
        $this->cpf = preg_replace('/[^0-9]/', '', $this->cpf);
        $this->registro_conselho = htmlspecialchars(strip_tags($this->registro_conselho));
        $this->tipo_registro = htmlspecialchars(strip_tags($this->tipo_registro));
        $this->salario = floatval($this->salario);
        $this->data_contratacao = htmlspecialchars(strip_tags($this->data_contratacao));
        $this->nome_arquivo_documento = htmlspecialchars(strip_tags($this->nome_arquivo_documento));
        $this->caminho_arquivo_documento = htmlspecialchars(strip_tags($this->caminho_arquivo_documento));
        $this->tamanho_mb_documento = floatval($this->tamanho_mb_documento);
        
        // Bind dos valores
        $stmt->bindParam(":nome_completo", $this->nome_completo);
        $stmt->bindParam(":cpf", $this->cpf);
        $stmt->bindParam(":registro_conselho", $this->registro_conselho);
        $stmt->bindParam(":tipo_registro", $this->tipo_registro);
        $stmt->bindParam(":salario", $this->salario);
        $stmt->bindParam(":data_contratacao", $this->data_contratacao);
        $stmt->bindParam(":nome_arquivo_documento", $this->nome_arquivo_documento);
        $stmt->bindParam(":caminho_arquivo_documento", $this->caminho_arquivo_documento);
        $stmt->bindParam(":tamanho_mb_documento", $this->tamanho_mb_documento);
        
        if($stmt->execute()) {
            $this->id_profissional = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    /**
     * Ler dados de um profissional
     */
    public function lerUm() {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id_profissional = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_profissional);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->nome_completo = $row['nome_completo'];
            $this->cpf = $row['cpf'];
            $this->registro_conselho = $row['registro_conselho'];
            $this->tipo_registro = $row['tipo_registro'];
            $this->salario = $row['salario'];
            $this->data_contratacao = $row['data_contratacao'];
            $this->nome_arquivo_documento = $row['nome_arquivo_documento'];
            $this->caminho_arquivo_documento = $row['caminho_arquivo_documento'];
            $this->tamanho_mb_documento = $row['tamanho_mb_documento'];
            $this->data_cadastro = $row['data_cadastro'];
            return true;
        }
        
        return false;
    }
    
    /**
     * Atualizar profissional
     */
    public function atualizar() {
        $query = "UPDATE " . $this->table_name . " 
                  SET nome_completo=:nome_completo, cpf=:cpf, registro_conselho=:registro_conselho, 
                      tipo_registro=:tipo_registro, salario=:salario, data_contratacao=:data_contratacao";
        
        // Se há novo documento, incluir na atualização
        if($this->nome_arquivo_documento) {
            $query .= ", nome_arquivo_documento=:nome_arquivo_documento, 
                         caminho_arquivo_documento=:caminho_arquivo_documento,
                         tamanho_mb_documento=:tamanho_mb_documento";
        }
        
        $query .= " WHERE id_profissional=:id_profissional";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->nome_completo = htmlspecialchars(strip_tags($this->nome_completo));
        $this->cpf = preg_replace('/[^0-9]/', '', $this->cpf);
        $this->registro_conselho = htmlspecialchars(strip_tags($this->registro_conselho));
        $this->tipo_registro = htmlspecialchars(strip_tags($this->tipo_registro));
        $this->salario = floatval($this->salario);
        $this->data_contratacao = htmlspecialchars(strip_tags($this->data_contratacao));
        $this->id_profissional = htmlspecialchars(strip_tags($this->id_profissional));
        
        // Bind dos valores básicos
        $stmt->bindParam(":nome_completo", $this->nome_completo);
        $stmt->bindParam(":cpf", $this->cpf);
        $stmt->bindParam(":registro_conselho", $this->registro_conselho);
        $stmt->bindParam(":tipo_registro", $this->tipo_registro);
        $stmt->bindParam(":salario", $this->salario);
        $stmt->bindParam(":data_contratacao", $this->data_contratacao);
        $stmt->bindParam(":id_profissional", $this->id_profissional);
        
        // Se há novo documento, fazer bind dos campos de documento
        if($this->nome_arquivo_documento) {
            $this->nome_arquivo_documento = htmlspecialchars(strip_tags($this->nome_arquivo_documento));
            $this->caminho_arquivo_documento = htmlspecialchars(strip_tags($this->caminho_arquivo_documento));
            $this->tamanho_mb_documento = floatval($this->tamanho_mb_documento);
            
            $stmt->bindParam(":nome_arquivo_documento", $this->nome_arquivo_documento);
            $stmt->bindParam(":caminho_arquivo_documento", $this->caminho_arquivo_documento);
            $stmt->bindParam(":tamanho_mb_documento", $this->tamanho_mb_documento);
        }
        
        return $stmt->execute();
    }
    
    /**
     * Deletar profissional
     */
    public function deletar() {
        // Primeiro, deletar o arquivo físico se existir
        if($this->caminho_arquivo_documento && file_exists($this->caminho_arquivo_documento)) {
            unlink($this->caminho_arquivo_documento);
        }
        
        $query = "DELETE FROM " . $this->table_name . " WHERE id_profissional = ?";
        
        $stmt = $this->conn->prepare($query);
        $this->id_profissional = htmlspecialchars(strip_tags($this->id_profissional));
        $stmt->bindParam(1, $this->id_profissional);
        
        return $stmt->execute();
    }
    
    /**
     * Listar todos os profissionais
     */
    public function listarTodos() {
        $query = "SELECT * FROM " . $this->table_name . " ORDER BY nome_completo ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Listar profissionais por tipo
     */
    public function listarPorTipo($tipo_registro) {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE tipo_registro = ? 
                  ORDER BY nome_completo ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $tipo_registro);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Verificar se CPF já existe
     */
    public function cpfExiste($cpf, $id_profissional = null) {
        $query = "SELECT id_profissional FROM " . $this->table_name . " WHERE cpf = ?";
        
        if($id_profissional) {
            $query .= " AND id_profissional != ?";
        }
        
        $stmt = $this->conn->prepare($query);
        $cpf = preg_replace('/[^0-9]/', '', $cpf);
        $stmt->bindParam(1, $cpf);
        
        if($id_profissional) {
            $stmt->bindParam(2, $id_profissional);
        }
        
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Verificar se registro do conselho já existe
     */
    public function registroExiste($registro_conselho, $id_profissional = null) {
        $query = "SELECT id_profissional FROM " . $this->table_name . " WHERE registro_conselho = ?";
        
        if($id_profissional) {
            $query .= " AND id_profissional != ?";
        }
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $registro_conselho);
        
        if($id_profissional) {
            $stmt->bindParam(2, $id_profissional);
        }
        
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Buscar profissionais por termo
     */
    public function buscar($termo) {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE nome_completo LIKE ? OR cpf LIKE ? OR registro_conselho LIKE ? 
                  ORDER BY nome_completo ASC";
        
        $stmt = $this->conn->prepare($query);
        $termo = "%{$termo}%";
        $stmt->bindParam(1, $termo);
        $stmt->bindParam(2, $termo);
        $stmt->bindParam(3, $termo);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Fazer upload do documento
     */
    public function fazerUploadDocumento($arquivo, $id_profissional) {
        // Validar arquivo
        $erros = validarUpload($arquivo);
        if(!empty($erros)) {
            return ['sucesso' => false, 'erros' => $erros];
        }
        
        // Criar diretório se não existir
        $diretorio = UPLOAD_PATH_PROFISSIONAIS . $id_profissional . '/';
        if(!is_dir($diretorio)) {
            mkdir($diretorio, 0755, true);
        }
        
        // Gerar nome único para o arquivo
        $extensao = pathinfo($arquivo['name'], PATHINFO_EXTENSION);
        $nome_arquivo = 'documento_' . time() . '.' . $extensao;
        $caminho_completo = $diretorio . $nome_arquivo;
        
        // Mover arquivo
        if(move_uploaded_file($arquivo['tmp_name'], $caminho_completo)) {
            // Calcular tamanho em MB
            $tamanho_mb = round($arquivo['size'] / (1024 * 1024), 2);
            
            return [
                'sucesso' => true, 
                'nome_arquivo' => $arquivo['name'],
                'caminho_arquivo' => $caminho_completo,
                'tamanho_mb' => $tamanho_mb
            ];
        } else {
            return ['sucesso' => false, 'erros' => ['Erro ao fazer upload do arquivo']];
        }
    }
    
    /**
     * Associar profissional a uma clínica
     */
    public function associarClinica($id_clinica) {
        $query = "INSERT IGNORE INTO clinica_profissional (id_clinica, id_profissional) VALUES (?, ?)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id_clinica);
        $stmt->bindParam(2, $this->id_profissional);
        
        return $stmt->execute();
    }
    
    /**
     * Desassociar profissional de uma clínica
     */
    public function desassociarClinica($id_clinica) {
        $query = "DELETE FROM clinica_profissional WHERE id_clinica = ? AND id_profissional = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id_clinica);
        $stmt->bindParam(2, $this->id_profissional);
        
        return $stmt->execute();
    }
    
    /**
     * Listar clínicas do profissional
     */
    public function listarClinicas() {
        $query = "SELECT c.* FROM clinicas c
                  INNER JOIN clinica_profissional cp ON c.id_clinica = cp.id_clinica
                  WHERE cp.id_profissional = ?
                  ORDER BY c.nome_clinica ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_profissional);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Contar total de profissionais
     */
    public function contarTotal() {
        $query = "SELECT COUNT(*) as total FROM " . $this->table_name;
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row['total'];
    }
    
    /**
     * Contar profissionais por tipo
     */
    public function contarPorTipo() {
        $query = "SELECT tipo_registro, COUNT(*) as total 
                  FROM " . $this->table_name . " 
                  GROUP BY tipo_registro";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>
