<?php
require_once '../../config/config.php';
require_once '../agendamentos/Agendamento.php';
require_once '../clinicas/Clinica.php';

verificarLogin();

$page_title = 'Pacientes Faltantes';

// Instanciar classes
$agendamento = new Agendamento();
$clinica = new Clinica();

// Obter datas selecionadas
$data_inicio = $_GET['data_inicio'] ?? date('Y-m-01'); // Primeiro dia do mês
$data_fim = $_GET['data_fim'] ?? date('Y-m-d'); // Hoje

// Obter clínica do usuário (se não for administrador)
$id_clinica_filtro = null;
if ($_SESSION['perfil_acesso'] !== 'Administrador') {
    $id_clinica_filtro = $_SESSION['id_clinica'];
} else {
    $id_clinica_filtro = $_GET['clinica'] ?? null;
}

// Obter dados
$pacientes_faltantes = [];

if ($id_clinica_filtro) {
    $stmt = $agendamento->listarFaltantes($data_inicio, $data_fim, $id_clinica_filtro);
    $pacientes_faltantes = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Obter lista de clínicas para administrador
$clinicas = [];
if ($_SESSION['perfil_acesso'] === 'Administrador') {
    $stmt_clinicas = $clinica->listarTodas();
    $clinicas = $stmt_clinicas->fetchAll(PDO::FETCH_ASSOC);
}

// Agrupar por paciente para estatísticas
$estatisticas_pacientes = [];
foreach ($pacientes_faltantes as $falta) {
    $cpf = $falta['cpf'];
    if (!isset($estatisticas_pacientes[$cpf])) {
        $estatisticas_pacientes[$cpf] = [
            'nome' => $falta['nome_paciente'],
            'telefone' => $falta['telefone_contato'],
            'total_faltas' => 0,
            'ultima_falta' => null,
            'faltas' => []
        ];
    }
    
    $estatisticas_pacientes[$cpf]['total_faltas']++;
    $estatisticas_pacientes[$cpf]['faltas'][] = $falta;
    
    if (!$estatisticas_pacientes[$cpf]['ultima_falta'] || 
        strtotime($falta['data_hora_consulta']) > strtotime($estatisticas_pacientes[$cpf]['ultima_falta'])) {
        $estatisticas_pacientes[$cpf]['ultima_falta'] = $falta['data_hora_consulta'];
    }
}

// Ordenar por número de faltas (decrescente)
uasort($estatisticas_pacientes, function($a, $b) {
    return $b['total_faltas'] - $a['total_faltas'];
});

include '../../includes/header.php';
?>

<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h1 class="h3 mb-0">
                <i class="fas fa-user-times me-2 text-danger"></i>
                Pacientes Faltantes
            </h1>
            <div class="d-flex gap-2">
                <a href="../agendamentos/agenda.php" class="btn btn-outline-secondary">
                    <i class="fas fa-calendar me-2"></i>
                    Ver Agenda
                </a>
                <button onclick="window.print()" class="btn btn-outline-primary">
                    <i class="fas fa-print me-2"></i>
                    Imprimir
                </button>
                <button onclick="exportarContatos()" class="btn btn-outline-success">
                    <i class="fas fa-phone me-2"></i>
                    Exportar Contatos
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Filtros -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card shadow">
            <div class="card-body">
                <form method="GET" action="">
                    <div class="row align-items-end">
                        <div class="col-md-3">
                            <label for="data_inicio" class="form-label">Data Início</label>
                            <input type="date" class="form-control" id="data_inicio" name="data_inicio" 
                                   value="<?php echo $data_inicio; ?>">
                        </div>
                        
                        <div class="col-md-3">
                            <label for="data_fim" class="form-label">Data Fim</label>
                            <input type="date" class="form-control" id="data_fim" name="data_fim" 
                                   value="<?php echo $data_fim; ?>">
                        </div>
                        
                        <?php if ($_SESSION['perfil_acesso'] === 'Administrador'): ?>
                        <div class="col-md-3">
                            <label for="clinica" class="form-label">Clínica</label>
                            <select class="form-select" id="clinica" name="clinica">
                                <option value="">Todas as clínicas</option>
                                <?php foreach ($clinicas as $clinica_item): ?>
                                    <option value="<?php echo $clinica_item['id_clinica']; ?>" 
                                            <?php echo $id_clinica_filtro == $clinica_item['id_clinica'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($clinica_item['nome_clinica']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <?php endif; ?>
                        
                        <div class="col-md-3">
                            <div class="d-grid">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-search me-2"></i>
                                    Consultar
                                </button>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<?php if ($id_clinica_filtro): ?>
<!-- Estatísticas Gerais -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card shadow">
            <div class="card-header bg-danger text-white">
                <h5 class="mb-0">
                    <i class="fas fa-chart-bar me-2"></i>
                    Estatísticas de Faltas - <?php echo date('d/m/Y', strtotime($data_inicio)); ?> a <?php echo date('d/m/Y', strtotime($data_fim)); ?>
                </h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <div class="text-center">
                            <h4 class="text-danger"><?php echo count($pacientes_faltantes); ?></h4>
                            <p class="text-muted mb-0">Total de Faltas</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center">
                            <h4 class="text-warning"><?php echo count($estatisticas_pacientes); ?></h4>
                            <p class="text-muted mb-0">Pacientes Diferentes</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center">
                            <?php 
                            $media_faltas = count($estatisticas_pacientes) > 0 ? 
                                round(count($pacientes_faltantes) / count($estatisticas_pacientes), 1) : 0;
                            ?>
                            <h4 class="text-info"><?php echo $media_faltas; ?></h4>
                            <p class="text-muted mb-0">Média Faltas/Paciente</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="text-center">
                            <?php 
                            $pacientes_com_contato = 0;
                            foreach ($estatisticas_pacientes as $pac) {
                                if (!empty($pac['telefone'])) $pacientes_com_contato++;
                            }
                            ?>
                            <h4 class="text-success"><?php echo $pacientes_com_contato; ?></h4>
                            <p class="text-muted mb-0">Com Contato</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Ranking de Pacientes Faltantes -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card shadow">
            <div class="card-header bg-warning text-dark">
                <h5 class="mb-0">
                    <i class="fas fa-trophy me-2"></i>
                    Ranking de Pacientes Faltantes
                </h5>
            </div>
            <div class="card-body">
                <?php if (empty($estatisticas_pacientes)): ?>
                    <div class="text-center py-4">
                        <i class="fas fa-smile fa-3x text-success mb-3"></i>
                        <h5 class="text-success">Excelente!</h5>
                        <p class="text-muted">
                            Não há pacientes faltantes no período selecionado.
                        </p>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Posição</th>
                                    <th>Paciente</th>
                                    <th>Contato</th>
                                    <th>Total de Faltas</th>
                                    <th>Última Falta</th>
                                    <th class="no-print">Ações</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $posicao = 1;
                                foreach ($estatisticas_pacientes as $cpf => $pac): 
                                    $classe_posicao = '';
                                    if ($posicao == 1) $classe_posicao = 'table-danger';
                                    elseif ($posicao <= 3) $classe_posicao = 'table-warning';
                                    elseif ($posicao <= 5) $classe_posicao = 'table-info';
                                ?>
                                <tr class="<?php echo $classe_posicao; ?>">
                                    <td>
                                        <strong><?php echo $posicao; ?>º</strong>
                                        <?php if ($posicao == 1): ?>
                                            <i class="fas fa-crown text-warning ms-1" title="Maior faltante"></i>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($pac['nome']); ?></strong>
                                        <br><small class="text-muted">
                                            CPF: <?php echo formatarCPF($cpf); ?>
                                        </small>
                                    </td>
                                    <td>
                                        <?php if (!empty($pac['telefone'])): ?>
                                            <span class="font-monospace">
                                                <?php echo formatarTelefone($pac['telefone']); ?>
                                            </span>
                                            <br>
                                            <a href="#" onclick="abrirWhatsApp('<?php echo $pac['telefone']; ?>', '<?php echo htmlspecialchars($pac['nome']); ?>')" 
                                               class="btn btn-sm whatsapp-btn mt-1">
                                                <i class="fab fa-whatsapp me-1"></i>
                                                WhatsApp
                                            </a>
                                        <?php else: ?>
                                            <span class="text-muted">Sem contato</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-danger fs-6">
                                            <?php echo $pac['total_faltas']; ?> falta<?php echo $pac['total_faltas'] > 1 ? 's' : ''; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php echo date('d/m/Y H:i', strtotime($pac['ultima_falta'])); ?>
                                        <br><small class="text-muted">
                                            <?php 
                                            $dias_atras = floor((time() - strtotime($pac['ultima_falta'])) / (60 * 60 * 24));
                                            echo $dias_atras . ' dia' . ($dias_atras != 1 ? 's' : '') . ' atrás';
                                            ?>
                                        </small>
                                    </td>
                                    <td class="no-print">
                                        <div class="btn-group btn-group-sm" role="group">
                                            <button type="button" class="btn btn-outline-info" 
                                                    onclick="verDetalhes('<?php echo $cpf; ?>')" 
                                                    title="Ver Detalhes">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                            
                                            <?php if (!empty($pac['telefone'])): ?>
                                                <button type="button" class="btn btn-outline-success" 
                                                        onclick="enviarLembrete('<?php echo $pac['telefone']; ?>', '<?php echo htmlspecialchars($pac['nome']); ?>')" 
                                                        title="Enviar Lembrete">
                                                    <i class="fas fa-bell"></i>
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php 
                                $posicao++;
                                endforeach; 
                                ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Lista Detalhada de Faltas -->
<div class="row">
    <div class="col-12">
        <div class="card shadow">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0">
                    <i class="fas fa-list me-2"></i>
                    Lista Detalhada de Faltas (<?php echo count($pacientes_faltantes); ?>)
                </h5>
            </div>
            <div class="card-body">
                <?php if (empty($pacientes_faltantes)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-smile fa-3x text-success mb-3"></i>
                        <h5 class="text-success">Excelente!</h5>
                        <p class="text-muted">
                            Não há faltas registradas no período selecionado.
                        </p>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Data/Hora</th>
                                    <th>Paciente</th>
                                    <th>Contato</th>
                                    <th>Tipo de Exame</th>
                                    <th>Profissional</th>
                                    <th>Clínica</th>
                                    <th class="no-print">Ações</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($pacientes_faltantes as $falta): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo date('d/m/Y', strtotime($falta['data_hora_consulta'])); ?></strong>
                                        <br><small class="text-muted">
                                            <?php echo date('H:i', strtotime($falta['data_hora_consulta'])); ?>
                                        </small>
                                    </td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($falta['nome_paciente']); ?></strong>
                                        <br><small class="text-muted">
                                            CPF: <?php echo formatarCPF($falta['cpf']); ?>
                                        </small>
                                    </td>
                                    <td>
                                        <?php if (!empty($falta['telefone_contato'])): ?>
                                            <span class="font-monospace">
                                                <?php echo formatarTelefone($falta['telefone_contato']); ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="text-muted">-</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-info">
                                            <?php echo htmlspecialchars($falta['tipo_exame_agendado']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if (!empty($falta['nome_profissional'])): ?>
                                            <strong><?php echo htmlspecialchars($falta['nome_profissional']); ?></strong>
                                        <?php else: ?>
                                            <span class="text-muted">Não definido</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <small><?php echo htmlspecialchars($falta['nome_clinica']); ?></small>
                                    </td>
                                    <td class="no-print">
                                        <div class="btn-group btn-group-sm" role="group">
                                            <a href="../agendamentos/visualizar.php?id=<?php echo $falta['id_agendamento']; ?>" 
                                               class="btn btn-outline-info" title="Ver Agendamento">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            
                                            <?php if (!empty($falta['telefone_contato'])): ?>
                                                <button type="button" class="btn btn-outline-success" 
                                                        onclick="abrirWhatsApp('<?php echo $falta['telefone_contato']; ?>', '<?php echo htmlspecialchars($falta['nome_paciente']); ?>')" 
                                                        title="WhatsApp">
                                                    <i class="fab fa-whatsapp"></i>
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php else: ?>
<div class="alert alert-warning" role="alert">
    <i class="fas fa-exclamation-triangle me-2"></i>
    <strong>Atenção!</strong> Selecione uma clínica para visualizar os pacientes faltantes.
</div>
<?php endif; ?>

<!-- Modal de Detalhes do Paciente -->
<div class="modal fade" id="modalDetalhes" tabindex="-1" aria-labelledby="modalDetalhesLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="modalDetalhesLabel">
                    <i class="fas fa-user me-2"></i>
                    Detalhes do Paciente
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="modalDetalhesBody">
                <!-- Conteúdo será carregado via JavaScript -->
            </div>
        </div>
    </div>
</div>

<style>
@media print {
    .no-print {
        display: none !important;
    }
    
    .card {
        border: none !important;
        box-shadow: none !important;
        page-break-inside: avoid;
    }
    
    .card-header {
        background-color: #f8f9fa !important;
        color: #000 !important;
        border-bottom: 2px solid #000 !important;
    }
    
    .btn {
        display: none !important;
    }
    
    .navbar {
        display: none !important;
    }
    
    .container-fluid {
        margin: 0 !important;
        padding: 0 !important;
    }
    
    .table {
        font-size: 11px;
    }
    
    .badge {
        border: 1px solid #000 !important;
        color: #000 !important;
        background-color: transparent !important;
    }
    
    .table-danger {
        background-color: #ffebee !important;
    }
    
    .table-warning {
        background-color: #fff8e1 !important;
    }
    
    .table-info {
        background-color: #e3f2fd !important;
    }
}
</style>

<script>
function abrirWhatsApp(telefone, nomePaciente) {
    const mensagem = `Olá ${nomePaciente}, notamos que você faltou em sua última consulta. Gostaríamos de reagendar. Entre em contato conosco.`;
    const url = gerarLinkWhatsApp(telefone, mensagem);
    window.open(url, '_blank');
}

function enviarLembrete(telefone, nomePaciente) {
    const mensagem = `Olá ${nomePaciente}, este é um lembrete sobre a importância de comparecer às consultas agendadas. Entre em contato para reagendar.`;
    const url = gerarLinkWhatsApp(telefone, mensagem);
    window.open(url, '_blank');
}

function verDetalhes(cpf) {
    // Buscar detalhes do paciente
    const paciente = <?php echo json_encode($estatisticas_pacientes); ?>[cpf];
    
    if (paciente) {
        let html = `
            <div class="row">
                <div class="col-md-6">
                    <h6>Informações do Paciente</h6>
                    <p><strong>Nome:</strong> ${paciente.nome}</p>
                    <p><strong>CPF:</strong> ${cpf}</p>
                    <p><strong>Telefone:</strong> ${paciente.telefone || 'Não informado'}</p>
                    <p><strong>Total de Faltas:</strong> ${paciente.total_faltas}</p>
                </div>
                <div class="col-md-6">
                    <h6>Histórico de Faltas</h6>
                    <div class="list-group">
        `;
        
        paciente.faltas.forEach(falta => {
            const data = new Date(falta.data_hora_consulta).toLocaleDateString('pt-BR');
            const hora = new Date(falta.data_hora_consulta).toLocaleTimeString('pt-BR', {hour: '2-digit', minute: '2-digit'});
            
            html += `
                <div class="list-group-item">
                    <div class="d-flex w-100 justify-content-between">
                        <h6 class="mb-1">${data} às ${hora}</h6>
                        <small class="text-muted">${falta.tipo_exame_agendado}</small>
                    </div>
                    <p class="mb-1">Profissional: ${falta.nome_profissional || 'Não definido'}</p>
                </div>
            `;
        });
        
        html += `
                    </div>
                </div>
            </div>
        `;
        
        document.getElementById('modalDetalhesBody').innerHTML = html;
        new bootstrap.Modal(document.getElementById('modalDetalhes')).show();
    }
}

function exportarContatos() {
    const pacientes = <?php echo json_encode($estatisticas_pacientes); ?>;
    let contatos = "Nome;CPF;Telefone;Total_Faltas\n";
    
    Object.keys(pacientes).forEach(cpf => {
        const pac = pacientes[cpf];
        contatos += `${pac.nome};${cpf};${pac.telefone || ''};${pac.total_faltas}\n`;
    });
    
    const blob = new Blob([contatos], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', 'pacientes_faltantes.csv');
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}
</script>

<?php include '../../includes/footer.php'; ?>
