<?php
require_once __DIR__ . '/../../config/database.php';

class TipoExame {
    private $conn;
    private $table_name = "tipos_exames";
    
    public $id_tipo_exame;
    public $nome_tipo_exame;
    public $valor_exame;
    public $descricao;
    public $ativo;
    public $data_criacao;
    
    public function __construct() {
        $this->conn = getDBConnection();
    }
    
    /**
     * Criar novo tipo de exame
     */
    public function criar() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET nome_tipo_exame=:nome_tipo_exame, valor_exame=:valor_exame,
                      descricao=:descricao, ativo=:ativo";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->nome_tipo_exame = htmlspecialchars(strip_tags($this->nome_tipo_exame));
        $this->valor_exame = floatval($this->valor_exame);
        $this->descricao = htmlspecialchars(strip_tags($this->descricao));
        $this->ativo = $this->ativo ? 1 : 0;
        
        // Bind dos valores
        $stmt->bindParam(":nome_tipo_exame", $this->nome_tipo_exame);
        $stmt->bindParam(":valor_exame", $this->valor_exame);
        $stmt->bindParam(":descricao", $this->descricao);
        $stmt->bindParam(":ativo", $this->ativo);
        
        if($stmt->execute()) {
            $this->id_tipo_exame = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    /**
     * Ler dados de um tipo de exame
     */
    public function lerUm() {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id_tipo_exame = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_tipo_exame);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->nome_tipo_exame = $row['nome_tipo_exame'];
            $this->valor_exame = $row['valor_exame'];
            $this->descricao = $row['descricao'];
            $this->ativo = $row['ativo'];
            $this->data_criacao = $row['data_criacao'];
            return $row;
        }
        
        return false;
    }
    
    /**
     * Atualizar tipo de exame
     */
    public function atualizar() {
        $query = "UPDATE " . $this->table_name . " 
                  SET nome_tipo_exame=:nome_tipo_exame, valor_exame=:valor_exame,
                      descricao=:descricao, ativo=:ativo
                  WHERE id_tipo_exame=:id_tipo_exame";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->nome_tipo_exame = htmlspecialchars(strip_tags($this->nome_tipo_exame));
        $this->valor_exame = floatval($this->valor_exame);
        $this->descricao = htmlspecialchars(strip_tags($this->descricao));
        $this->ativo = $this->ativo ? 1 : 0;
        $this->id_tipo_exame = htmlspecialchars(strip_tags($this->id_tipo_exame));
        
        // Bind dos valores
        $stmt->bindParam(":nome_tipo_exame", $this->nome_tipo_exame);
        $stmt->bindParam(":valor_exame", $this->valor_exame);
        $stmt->bindParam(":descricao", $this->descricao);
        $stmt->bindParam(":ativo", $this->ativo);
        $stmt->bindParam(":id_tipo_exame", $this->id_tipo_exame);
        
        return $stmt->execute();
    }
    
    /**
     * Deletar tipo de exame
     */
    public function deletar() {
        // Verificar se há agendamentos usando este tipo de exame
        $query_check = "SELECT COUNT(*) as total FROM agendamentos WHERE tipo_exame_agendado = ?";
        $stmt_check = $this->conn->prepare($query_check);
        $stmt_check->bindParam(1, $this->nome_tipo_exame);
        $stmt_check->execute();
        $result = $stmt_check->fetch(PDO::FETCH_ASSOC);
        
        if ($result['total'] > 0) {
            return false; // Não pode deletar se há agendamentos
        }
        
        $query = "DELETE FROM " . $this->table_name . " WHERE id_tipo_exame = ?";
        
        $stmt = $this->conn->prepare($query);
        $this->id_tipo_exame = htmlspecialchars(strip_tags($this->id_tipo_exame));
        $stmt->bindParam(1, $this->id_tipo_exame);
        
        return $stmt->execute();
    }
    
    /**
     * Listar todos os tipos de exames
     */
    public function listarTodos($apenas_ativos = false) {
        $where = "";
        
        if($apenas_ativos) {
            $where = "WHERE ativo = 1";
        }
        
        $query = "SELECT * FROM " . $this->table_name . " 
                  " . $where . "
                  ORDER BY nome_tipo_exame ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Verificar se nome já existe
     */
    public function nomeExiste($nome, $id_excluir = null) {
        $query = "SELECT id_tipo_exame FROM " . $this->table_name . " WHERE nome_tipo_exame = ?";
        
        if($id_excluir) {
            $query .= " AND id_tipo_exame != ?";
        }
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $nome);
        
        if($id_excluir) {
            $stmt->bindParam(2, $id_excluir);
        }
        
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Ativar/Desativar tipo de exame
     */
    public function alterarStatus($ativo) {
        $query = "UPDATE " . $this->table_name . " 
                  SET ativo = ? 
                  WHERE id_tipo_exame = ?";
        
        $stmt = $this->conn->prepare($query);
        $ativo_int = $ativo ? 1 : 0;
        $stmt->bindParam(1, $ativo_int);
        $stmt->bindParam(2, $this->id_tipo_exame);
        
        return $stmt->execute();
    }
    
    /**
     * Obter estatísticas de tipos de exames
     */
    public function obterEstatisticas() {
        $query = "SELECT 
                    COUNT(*) as total_tipos,
                    COUNT(CASE WHEN ativo = 1 THEN 1 END) as tipos_ativos,
                    COUNT(CASE WHEN ativo = 0 THEN 1 END) as tipos_inativos,
                    AVG(valor_exame) as valor_medio,
                    MIN(valor_exame) as valor_minimo,
                    MAX(valor_exame) as valor_maximo
                  FROM " . $this->table_name;
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Buscar tipos de exames
     */
    public function buscar($termo) {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE nome_tipo_exame LIKE ? OR descricao LIKE ?
                  ORDER BY nome_tipo_exame ASC";
        
        $stmt = $this->conn->prepare($query);
        $termo_busca = "%{$termo}%";
        $stmt->bindParam(1, $termo_busca);
        $stmt->bindParam(2, $termo_busca);
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Obter tipos mais utilizados
     */
    public function obterMaisUtilizados($limite = 5) {
        $query = "SELECT te.*, COUNT(a.id_agendamento) as total_agendamentos
                  FROM " . $this->table_name . " te
                  LEFT JOIN agendamentos a ON te.nome_tipo_exame = a.tipo_exame_agendado
                  WHERE te.ativo = 1
                  GROUP BY te.id_tipo_exame
                  ORDER BY total_agendamentos DESC, te.nome_tipo_exame ASC
                  LIMIT ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $limite, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter valor por nome do tipo
     */
    public function obterValorPorNome($nome_tipo) {
        $query = "SELECT valor_exame FROM " . $this->table_name . " 
                  WHERE nome_tipo_exame = ? AND ativo = 1 LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $nome_tipo);
        $stmt->execute();
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $result ? $result['valor_exame'] : 0;
    }
    
    /**
     * Listar para select (id => nome)
     */
    public function listarParaSelect($apenas_ativos = true) {
        $where = "";
        
        if($apenas_ativos) {
            $where = "WHERE ativo = 1";
        }
        
        $query = "SELECT id_tipo_exame, nome_tipo_exame, valor_exame 
                  FROM " . $this->table_name . " 
                  " . $where . "
                  ORDER BY nome_tipo_exame ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        $tipos = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $tipos[$row['id_tipo_exame']] = [
                'nome' => $row['nome_tipo_exame'],
                'valor' => $row['valor_exame']
            ];
        }
        
        return $tipos;
    }
    
    /**
     * Verificar se pode ser deletado
     */
    public function podeSerDeletado() {
        // Verificar agendamentos
        $query = "SELECT COUNT(*) as total FROM agendamentos WHERE tipo_exame_agendado = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->nome_tipo_exame);
        $stmt->execute();
        $agendamentos = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Verificar atendimentos
        $query = "SELECT COUNT(*) as total FROM atendimentos WHERE tipo_exame_realizado = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->nome_tipo_exame);
        $stmt->execute();
        $atendimentos = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return ($agendamentos['total'] == 0 && $atendimentos['total'] == 0);
    }
    
    /**
     * Obter uso do tipo de exame
     */
    public function obterUso() {
        $uso = [
            'agendamentos' => 0,
            'atendimentos' => 0,
            'total' => 0
        ];
        
        // Contar agendamentos
        $query = "SELECT COUNT(*) as total FROM agendamentos WHERE tipo_exame_agendado = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->nome_tipo_exame);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $uso['agendamentos'] = $result['total'];
        
        // Contar atendimentos
        $query = "SELECT COUNT(*) as total FROM atendimentos WHERE tipo_exame_realizado = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->nome_tipo_exame);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $uso['atendimentos'] = $result['total'];
        
        $uso['total'] = $uso['agendamentos'] + $uso['atendimentos'];
        
        return $uso;
    }
}
?>
