<?php
require_once __DIR__ . '/../../config/database.php';

class Usuario {
    private $conn;
    private $table_name = "usuarios";
    
    public $id_usuario;
    public $id_clinica;
    public $nome_usuario;
    public $email;
    public $senha;
    public $perfil_acesso;
    public $ativo;
    public $ultimo_login;
    public $data_criacao;
    
    public function __construct() {
        $this->conn = getDBConnection();
    }
    
    /**
     * Criar novo usuário
     */
    public function criar() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET id_clinica=:id_clinica, nome_usuario=:nome_usuario, email=:email,
                      senha=:senha, perfil_acesso=:perfil_acesso, ativo=:ativo";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->id_clinica = htmlspecialchars(strip_tags($this->id_clinica));
        $this->nome_usuario = htmlspecialchars(strip_tags($this->nome_usuario));
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->senha = password_hash($this->senha, PASSWORD_DEFAULT);
        $this->perfil_acesso = htmlspecialchars(strip_tags($this->perfil_acesso));
        $this->ativo = $this->ativo ? 1 : 0;
        
        // Bind dos valores
        $stmt->bindParam(":id_clinica", $this->id_clinica);
        $stmt->bindParam(":nome_usuario", $this->nome_usuario);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":senha", $this->senha);
        $stmt->bindParam(":perfil_acesso", $this->perfil_acesso);
        $stmt->bindParam(":ativo", $this->ativo);
        
        if($stmt->execute()) {
            $this->id_usuario = $this->conn->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    /**
     * Ler dados de um usuário
     */
    public function lerUm() {
        $query = "SELECT u.*, c.nome_clinica 
                  FROM " . $this->table_name . " u
                  LEFT JOIN clinicas c ON u.id_clinica = c.id_clinica
                  WHERE u.id_usuario = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_usuario);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row) {
            $this->id_clinica = $row['id_clinica'];
            $this->nome_usuario = $row['nome_usuario'];
            $this->email = $row['email'];
            $this->perfil_acesso = $row['perfil_acesso'];
            $this->ativo = $row['ativo'];
            $this->ultimo_login = $row['ultimo_login'];
            $this->data_criacao = $row['data_criacao'];
            return $row;
        }
        
        return false;
    }
    
    /**
     * Atualizar usuário
     */
    public function atualizar() {
        $query = "UPDATE " . $this->table_name . " 
                  SET id_clinica=:id_clinica, nome_usuario=:nome_usuario, email=:email,
                      perfil_acesso=:perfil_acesso, ativo=:ativo";
        
        // Se senha foi fornecida, incluir na atualização
        if (!empty($this->senha)) {
            $query .= ", senha=:senha";
        }
        
        $query .= " WHERE id_usuario=:id_usuario";
        
        $stmt = $this->conn->prepare($query);
        
        // Limpar dados
        $this->id_clinica = htmlspecialchars(strip_tags($this->id_clinica));
        $this->nome_usuario = htmlspecialchars(strip_tags($this->nome_usuario));
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->perfil_acesso = htmlspecialchars(strip_tags($this->perfil_acesso));
        $this->ativo = $this->ativo ? 1 : 0;
        $this->id_usuario = htmlspecialchars(strip_tags($this->id_usuario));
        
        // Bind dos valores
        $stmt->bindParam(":id_clinica", $this->id_clinica);
        $stmt->bindParam(":nome_usuario", $this->nome_usuario);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":perfil_acesso", $this->perfil_acesso);
        $stmt->bindParam(":ativo", $this->ativo);
        $stmt->bindParam(":id_usuario", $this->id_usuario);
        
        if (!empty($this->senha)) {
            $senha_hash = password_hash($this->senha, PASSWORD_DEFAULT);
            $stmt->bindParam(":senha", $senha_hash);
        }
        
        return $stmt->execute();
    }
    
    /**
     * Deletar usuário
     */
    public function deletar() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id_usuario = ?";
        
        $stmt = $this->conn->prepare($query);
        $this->id_usuario = htmlspecialchars(strip_tags($this->id_usuario));
        $stmt->bindParam(1, $this->id_usuario);
        
        return $stmt->execute();
    }
    
    /**
     * Listar todos os usuários
     */
    public function listarTodos($id_clinica = null) {
        $where = "";
        
        if($id_clinica) {
            $where = "WHERE u.id_clinica = ?";
        }
        
        $query = "SELECT u.*, c.nome_clinica 
                  FROM " . $this->table_name . " u
                  LEFT JOIN clinicas c ON u.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY u.nome_usuario ASC";
        
        $stmt = $this->conn->prepare($query);
        
        if($id_clinica) {
            $stmt->bindParam(1, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Autenticar usuário
     */
    public function autenticar($email, $senha) {
        $query = "SELECT u.*, c.nome_clinica 
                  FROM " . $this->table_name . " u
                  LEFT JOIN clinicas c ON u.id_clinica = c.id_clinica
                  WHERE u.email = ? AND u.ativo = 1 LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $email);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row && password_verify($senha, $row['senha'])) {
            // Atualizar último login
            $this->atualizarUltimoLogin($row['id_usuario']);
            
            return $row;
        }
        
        return false;
    }
    
    /**
     * Atualizar último login
     */
    private function atualizarUltimoLogin($id_usuario) {
        $query = "UPDATE " . $this->table_name . " 
                  SET ultimo_login = NOW() 
                  WHERE id_usuario = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $id_usuario);
        
        return $stmt->execute();
    }
    
    /**
     * Verificar se email já existe
     */
    public function emailExiste($email, $id_usuario_excluir = null) {
        $query = "SELECT id_usuario FROM " . $this->table_name . " WHERE email = ?";
        
        if($id_usuario_excluir) {
            $query .= " AND id_usuario != ?";
        }
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $email);
        
        if($id_usuario_excluir) {
            $stmt->bindParam(2, $id_usuario_excluir);
        }
        
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }
    
    /**
     * Alterar senha
     */
    public function alterarSenha($senha_atual, $senha_nova) {
        // Primeiro verificar a senha atual
        $query = "SELECT senha FROM " . $this->table_name . " WHERE id_usuario = ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id_usuario);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($row && password_verify($senha_atual, $row['senha'])) {
            // Atualizar com a nova senha
            $query = "UPDATE " . $this->table_name . " 
                      SET senha = ? 
                      WHERE id_usuario = ?";
            
            $stmt = $this->conn->prepare($query);
            $senha_hash = password_hash($senha_nova, PASSWORD_DEFAULT);
            $stmt->bindParam(1, $senha_hash);
            $stmt->bindParam(2, $this->id_usuario);
            
            return $stmt->execute();
        }
        
        return false;
    }
    
    /**
     * Ativar/Desativar usuário
     */
    public function alterarStatus($ativo) {
        $query = "UPDATE " . $this->table_name . " 
                  SET ativo = ? 
                  WHERE id_usuario = ?";
        
        $stmt = $this->conn->prepare($query);
        $ativo_int = $ativo ? 1 : 0;
        $stmt->bindParam(1, $ativo_int);
        $stmt->bindParam(2, $this->id_usuario);
        
        return $stmt->execute();
    }
    
    /**
     * Obter usuários por perfil
     */
    public function listarPorPerfil($perfil, $id_clinica = null) {
        $where = "WHERE u.perfil_acesso = ?";
        
        if($id_clinica) {
            $where .= " AND u.id_clinica = ?";
        }
        
        $query = "SELECT u.*, c.nome_clinica 
                  FROM " . $this->table_name . " u
                  LEFT JOIN clinicas c ON u.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY u.nome_usuario ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $perfil);
        
        if($id_clinica) {
            $stmt->bindParam(2, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
    
    /**
     * Obter estatísticas de usuários
     */
    public function obterEstatisticas($id_clinica = null) {
        $where = "";
        
        if($id_clinica) {
            $where = "WHERE id_clinica = ?";
        }
        
        $query = "SELECT 
                    COUNT(*) as total_usuarios,
                    COUNT(CASE WHEN ativo = 1 THEN 1 END) as usuarios_ativos,
                    COUNT(CASE WHEN ativo = 0 THEN 1 END) as usuarios_inativos,
                    COUNT(CASE WHEN perfil_acesso = 'Administrador' THEN 1 END) as administradores,
                    COUNT(CASE WHEN perfil_acesso = 'Gerente' THEN 1 END) as gerentes,
                    COUNT(CASE WHEN perfil_acesso = 'Secretaria' THEN 1 END) as secretarias,
                    COUNT(CASE WHEN ultimo_login >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 END) as ativos_ultimo_mes
                  FROM " . $this->table_name . " 
                  " . $where;
        
        $stmt = $this->conn->prepare($query);
        
        if($id_clinica) {
            $stmt->bindParam(1, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Obter usuários recentes
     */
    public function obterUsuariosRecentes($limite = 5, $id_clinica = null) {
        $where = "";
        
        if($id_clinica) {
            $where = "WHERE u.id_clinica = ?";
        }
        
        $query = "SELECT u.*, c.nome_clinica 
                  FROM " . $this->table_name . " u
                  LEFT JOIN clinicas c ON u.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY u.data_criacao DESC
                  LIMIT ?";
        
        $stmt = $this->conn->prepare($query);
        
        $param_index = 1;
        if($id_clinica) {
            $stmt->bindParam($param_index++, $id_clinica);
        }
        
        $stmt->bindParam($param_index, $limite, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Verificar permissão de acesso
     */
    public function verificarPermissao($perfil_necessario, $perfil_usuario) {
        $hierarquia = [
            'Secretaria' => 1,
            'Gerente' => 2,
            'Administrador' => 3
        ];
        
        $nivel_necessario = $hierarquia[$perfil_necessario] ?? 0;
        $nivel_usuario = $hierarquia[$perfil_usuario] ?? 0;
        
        return $nivel_usuario >= $nivel_necessario;
    }
    
    /**
     * Resetar senha (gerar nova senha temporária)
     */
    public function resetarSenha() {
        // Gerar senha temporária
        $senha_temporaria = substr(str_shuffle('0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'), 0, 8);
        
        $query = "UPDATE " . $this->table_name . " 
                  SET senha = ? 
                  WHERE id_usuario = ?";
        
        $stmt = $this->conn->prepare($query);
        $senha_hash = password_hash($senha_temporaria, PASSWORD_DEFAULT);
        $stmt->bindParam(1, $senha_hash);
        $stmt->bindParam(2, $this->id_usuario);
        
        if($stmt->execute()) {
            return $senha_temporaria;
        }
        
        return false;
    }
    
    /**
     * Buscar usuários
     */
    public function buscar($termo, $id_clinica = null) {
        $where = "WHERE (u.nome_usuario LIKE ? OR u.email LIKE ?)";
        
        if($id_clinica) {
            $where .= " AND u.id_clinica = ?";
        }
        
        $query = "SELECT u.*, c.nome_clinica 
                  FROM " . $this->table_name . " u
                  LEFT JOIN clinicas c ON u.id_clinica = c.id_clinica
                  " . $where . "
                  ORDER BY u.nome_usuario ASC";
        
        $stmt = $this->conn->prepare($query);
        $termo_busca = "%{$termo}%";
        $stmt->bindParam(1, $termo_busca);
        $stmt->bindParam(2, $termo_busca);
        
        if($id_clinica) {
            $stmt->bindParam(3, $id_clinica);
        }
        
        $stmt->execute();
        
        return $stmt;
    }
}
?>
