<?php
/**
 * Teste de Funcionalidades do Sistema
 * Execute este arquivo para verificar se todas as funcionalidades estão funcionando
 */

// Verificar se o sistema foi instalado
if (!file_exists('config/installed.lock')) {
    die('Sistema não foi instalado. Execute install.php primeiro.');
}

require_once 'config/config.php';

$testes = [];
$erros = [];

// Teste 1: Conexão com banco de dados
try {
    $conn = getDBConnection();
    $testes['database'] = '✅ Conexão com banco de dados';
} catch (Exception $e) {
    $erros['database'] = '❌ Erro na conexão: ' . $e->getMessage();
}

// Teste 2: Verificar tabelas
try {
    $tabelas_necessarias = [
        'usuarios', 'clinicas', 'profissionais', 'pacientes', 
        'agendamentos', 'atendimentos', 'tipos_exames', 
        'documentos_clinicas', 'despesas_diarias', 'fechamentos_diarios'
    ];
    
    $stmt = $conn->query("SHOW TABLES");
    $tabelas_existentes = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    $tabelas_faltando = array_diff($tabelas_necessarias, $tabelas_existentes);
    
    if (empty($tabelas_faltando)) {
        $testes['tables'] = '✅ Todas as tabelas estão criadas';
    } else {
        $erros['tables'] = '❌ Tabelas faltando: ' . implode(', ', $tabelas_faltando);
    }
} catch (Exception $e) {
    $erros['tables'] = '❌ Erro ao verificar tabelas: ' . $e->getMessage();
}

// Teste 3: Verificar usuário administrador
try {
    $stmt = $conn->query("SELECT COUNT(*) FROM usuarios WHERE perfil_acesso = 'Administrador'");
    $admin_count = $stmt->fetchColumn();
    
    if ($admin_count > 0) {
        $testes['admin'] = '✅ Usuário administrador existe';
    } else {
        $erros['admin'] = '❌ Nenhum usuário administrador encontrado';
    }
} catch (Exception $e) {
    $erros['admin'] = '❌ Erro ao verificar administrador: ' . $e->getMessage();
}

// Teste 4: Verificar permissões de diretórios
$diretorios = ['uploads', 'logs', 'config'];
foreach ($diretorios as $dir) {
    if (is_dir($dir) && is_writable($dir)) {
        $testes["dir_$dir"] = "✅ Diretório $dir com permissões corretas";
    } else {
        $erros["dir_$dir"] = "❌ Diretório $dir sem permissões de escrita";
    }
}

// Teste 5: Verificar extensões PHP
$extensoes_necessarias = ['pdo', 'pdo_mysql', 'mbstring', 'fileinfo', 'json'];
foreach ($extensoes_necessarias as $ext) {
    if (extension_loaded($ext)) {
        $testes["ext_$ext"] = "✅ Extensão $ext carregada";
    } else {
        $erros["ext_$ext"] = "❌ Extensão $ext não encontrada";
    }
}

// Teste 6: Verificar configurações PHP
$configs = [
    'upload_max_filesize' => '5M',
    'post_max_size' => '5M',
    'max_execution_time' => 300
];

foreach ($configs as $config => $valor_esperado) {
    $valor_atual = ini_get($config);
    if ($config === 'upload_max_filesize' || $config === 'post_max_size') {
        $valor_atual_bytes = return_bytes($valor_atual);
        $valor_esperado_bytes = return_bytes($valor_esperado);
        if ($valor_atual_bytes >= $valor_esperado_bytes) {
            $testes["config_$config"] = "✅ $config: $valor_atual (OK)";
        } else {
            $erros["config_$config"] = "❌ $config: $valor_atual (esperado: $valor_esperado)";
        }
    } else {
        if ($valor_atual >= $valor_esperado) {
            $testes["config_$config"] = "✅ $config: $valor_atual (OK)";
        } else {
            $erros["config_$config"] = "❌ $config: $valor_atual (esperado: $valor_esperado)";
        }
    }
}

// Teste 7: Verificar classes principais
$classes = [
    'modules/usuarios/Usuario.php' => 'Usuario',
    'modules/clinicas/Clinica.php' => 'Clinica',
    'modules/profissionais/Profissional.php' => 'Profissional',
    'modules/agendamentos/Agendamento.php' => 'Agendamento',
    'modules/atendimentos/Atendimento.php' => 'Atendimento'
];

foreach ($classes as $arquivo => $classe) {
    if (file_exists($arquivo)) {
        require_once $arquivo;
        if (class_exists($classe)) {
            $testes["class_$classe"] = "✅ Classe $classe carregada";
        } else {
            $erros["class_$classe"] = "❌ Classe $classe não encontrada no arquivo";
        }
    } else {
        $erros["class_$classe"] = "❌ Arquivo $arquivo não encontrado";
    }
}

function return_bytes($val) {
    $val = trim($val);
    $last = strtolower($val[strlen($val)-1]);
    $val = (int) $val;
    switch($last) {
        case 'g':
            $val *= 1024;
        case 'm':
            $val *= 1024;
        case 'k':
            $val *= 1024;
    }
    return $val;
}

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teste do Sistema - Clínicas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .test-container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .test-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .test-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .test-item {
            padding: 10px 0;
            border-bottom: 1px solid #eee;
        }
        
        .test-item:last-child {
            border-bottom: none;
        }
        
        .status-ok {
            color: #28a745;
        }
        
        .status-error {
            color: #dc3545;
        }
        
        .summary-card {
            border-radius: 10px;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="test-container">
        <div class="test-card">
            <div class="test-header">
                <h1><i class="fas fa-vial me-3"></i>Teste do Sistema</h1>
                <p class="mb-0">Verificação de Funcionalidades e Configurações</p>
            </div>
            
            <div class="p-4">
                <!-- Resumo -->
                <div class="row mb-4">
                    <div class="col-md-6">
                        <div class="card summary-card bg-success text-white">
                            <div class="card-body text-center">
                                <h3><?php echo count($testes); ?></h3>
                                <p class="mb-0">Testes Aprovados</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card summary-card bg-danger text-white">
                            <div class="card-body text-center">
                                <h3><?php echo count($erros); ?></h3>
                                <p class="mb-0">Testes com Erro</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Status Geral -->
                <?php if (empty($erros)): ?>
                    <div class="alert alert-success" role="alert">
                        <h4 class="alert-heading"><i class="fas fa-check-circle me-2"></i>Sistema Funcionando Perfeitamente!</h4>
                        <p>Todos os testes foram aprovados. O sistema está pronto para uso.</p>
                        <hr>
                        <p class="mb-0">
                            <a href="login.php" class="btn btn-success me-2">
                                <i class="fas fa-sign-in-alt me-2"></i>Acessar Sistema
                            </a>
                            <a href="dashboard.php" class="btn btn-outline-success">
                                <i class="fas fa-tachometer-alt me-2"></i>Dashboard
                            </a>
                        </p>
                    </div>
                <?php else: ?>
                    <div class="alert alert-warning" role="alert">
                        <h4 class="alert-heading"><i class="fas fa-exclamation-triangle me-2"></i>Atenção!</h4>
                        <p>Alguns testes falharam. Verifique os erros abaixo antes de usar o sistema.</p>
                    </div>
                <?php endif; ?>
                
                <!-- Testes Aprovados -->
                <?php if (!empty($testes)): ?>
                    <div class="card mb-4">
                        <div class="card-header bg-success text-white">
                            <h5 class="mb-0">
                                <i class="fas fa-check-circle me-2"></i>
                                Testes Aprovados (<?php echo count($testes); ?>)
                            </h5>
                        </div>
                        <div class="card-body">
                            <?php foreach ($testes as $teste => $mensagem): ?>
                                <div class="test-item status-ok">
                                    <?php echo $mensagem; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>
                
                <!-- Erros Encontrados -->
                <?php if (!empty($erros)): ?>
                    <div class="card mb-4">
                        <div class="card-header bg-danger text-white">
                            <h5 class="mb-0">
                                <i class="fas fa-exclamation-triangle me-2"></i>
                                Erros Encontrados (<?php echo count($erros); ?>)
                            </h5>
                        </div>
                        <div class="card-body">
                            <?php foreach ($erros as $erro => $mensagem): ?>
                                <div class="test-item status-error">
                                    <?php echo $mensagem; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>
                
                <!-- Informações do Sistema -->
                <div class="card">
                    <div class="card-header bg-info text-white">
                        <h5 class="mb-0">
                            <i class="fas fa-info-circle me-2"></i>
                            Informações do Sistema
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <p><strong>Versão PHP:</strong> <?php echo PHP_VERSION; ?></p>
                                <p><strong>Servidor Web:</strong> <?php echo $_SERVER['SERVER_SOFTWARE'] ?? 'Não identificado'; ?></p>
                                <p><strong>Sistema Operacional:</strong> <?php echo PHP_OS; ?></p>
                            </div>
                            <div class="col-md-6">
                                <p><strong>Memória Limite:</strong> <?php echo ini_get('memory_limit'); ?></p>
                                <p><strong>Upload Máximo:</strong> <?php echo ini_get('upload_max_filesize'); ?></p>
                                <p><strong>Tempo Execução:</strong> <?php echo ini_get('max_execution_time'); ?>s</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Ações -->
                <div class="text-center mt-4">
                    <button onclick="location.reload()" class="btn btn-primary me-2">
                        <i class="fas fa-redo me-2"></i>Executar Testes Novamente
                    </button>
                    
                    <?php if (empty($erros)): ?>
                        <a href="login.php" class="btn btn-success">
                            <i class="fas fa-sign-in-alt me-2"></i>Acessar Sistema
                        </a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- Informações Adicionais -->
        <div class="text-center mt-4">
            <small class="text-white-50">
                Sistema de Controle e Administração Financeira de Clínicas<br>
                Teste executado em <?php echo date('d/m/Y H:i:s'); ?>
            </small>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
